package com.walker.jdbc.mongo;

import com.mongodb.client.MongoCollection;
import com.mongodb.client.result.DeleteResult;
import com.mongodb.client.result.UpdateResult;
import com.walker.db.Sorts;
import com.walker.db.page.GenericPager;
import com.walker.db.page.ListPageContext;
import com.walker.db.page.PageSearch;
import com.walker.infrastructure.utils.StringUtils;
import org.bson.Document;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.lang.Nullable;

import java.util.List;

/**
 * Mongo数据库操作服务基础封装。
 * @author 时克英
 * @date 2023-07-06
 */
public class MongoService {

    public <T> MongoCollection createCollection(Class<T> entity){
        return this.mongoTemplate.createCollection(entity);
    }

    public void createCollection(String name){
        this.mongoTemplate.createCollection(name);
    }

    public <T> void dropCollection(Class<T> entity){
        this.mongoTemplate.dropCollection(entity);
    }
    public void dropCollection(String tableName){
        this.mongoTemplate.dropCollection(tableName);
    }

    public <T> void insert(T data, String tableName){
        this.mongoTemplate.insert(data, tableName);
    }

    public <T> void insert(T data){
        this.mongoTemplate.insert(data);
    }

    public <T> void insertBatch(List<T> data, Class<T> entity){
        this.mongoTemplate.insert(data, entity);
    }

    /**
     * 适合批量写入，表明是动态的，例如：聊天中每个人一张表。
     * @param data 写入记录集合
     * @param tableName 表名
     * @param <T>
     * @date 2023-07-06
     */
    public <T> void insertBatch(List<T> data, String tableName){
        this.mongoTemplate.insert(data, tableName);
    }

    public long deleteById(String id, String tableName){
        Query query = new Query(Criteria.where(KEY_ID_DEFAULT).is(id));
        DeleteResult deleteResult = this.mongoTemplate.remove(query, tableName);
        if(deleteResult != null){
            return deleteResult.getDeletedCount();
        }
        return 0;
    }
    public <T> long deleteById(String id, Class<T> entity){
        Query query = new Query(Criteria.where(KEY_ID_DEFAULT).is(id));
        DeleteResult deleteResult = this.mongoTemplate.remove(query, entity);
        if(deleteResult != null){
            return deleteResult.getDeletedCount();
        }
        return 0;
    }

    /**
     * 更新一条记录
     * @param id 主键，即：对应表中默认字段_id的值
     * @param column 更新字段名称
     * @param value 更新值
     * @param tableName 表名
     * @return
     * @date 2023-07-06
     */
    public UpdateResult updateById(String id, String column, Object value, String tableName){
        Query query = new Query(Criteria.where(KEY_ID_DEFAULT).is(id));
        Update update = new Update();
        update.set(column, value);
        UpdateResult updateResult = this.mongoTemplate.updateFirst(query, update, tableName);
        return updateResult;
    }

    public <T> UpdateResult updateById(String id, String column, Object value, Class<T> entity){
        Query query = new Query(Criteria.where(KEY_ID_DEFAULT).is(id));
        Update update = new Update();
        update.set(column, value);
        UpdateResult updateResult = this.mongoTemplate.updateFirst(query, update, entity);
        return updateResult;
    }

    /**
     * 更新一个完整对象（记录）。
     * @param id
     * @param data
     * @return
     * @param <T>
     */
    public <T> UpdateResult updateEntity(String id, T data){
        Query query = new Query(Criteria.where(KEY_ID_DEFAULT).is(id));
        Document document = (Document)mongoTemplate.getConverter().convertToMongoType(data);
        Update update = Update.fromDocument(document);
        UpdateResult updateResult = this.mongoTemplate.updateFirst(query, update, data.getClass());
        return updateResult;
    }

    public <T> UpdateResult updateEntity(String id, T data, String tableName){
        Query query = new Query(Criteria.where(KEY_ID_DEFAULT).is(id));
        Document document = (Document)mongoTemplate.getConverter().convertToMongoType(data);
        Update update = Update.fromDocument(document);
        UpdateResult updateResult = this.mongoTemplate.updateFirst(query, update, tableName);
        return updateResult;
    }

    public <T> List<T> queryList(Query query, Class<T> entity, @Nullable String tableName, @Nullable Sorts.Sort sorts){
        if(sorts != null){
            if(sorts.isAsc()){
                query.with(Sort.by(Sort.Order.asc(sorts.getField())));
            } else {
                query.with(Sort.by(Sort.Order.desc(sorts.getField())));
            }
        }
        List<T> data = null;
        if(StringUtils.isEmpty(tableName)){
            data = this.mongoTemplate.find(query, entity);
        } else {
            data = this.mongoTemplate.find(query, entity, tableName);
        }
        return data;
    }

    public <T> GenericPager<T> queryPageList(Query query, Class<T> entity, @Nullable String tableName
            , PageSearch pageSearch, @Nullable Sorts.Sort sorts){

        query.with(PageRequest.of(pageSearch.getPageIndex()-1, pageSearch.getPageSize()));
        if(sorts != null){
            if(sorts.isAsc()){
                query.with(Sort.by(Sort.Order.asc(sorts.getField())));
            } else {
                query.with(Sort.by(Sort.Order.desc(sorts.getField())));
            }
        }
        long total = 0;
        List<T> data = null;
        if(StringUtils.isEmpty(tableName)){
            total = this.mongoTemplate.count(query, entity);
            data = this.mongoTemplate.find(query, entity);
        } else {
            total = this.mongoTemplate.count(query, entity, tableName);
            data = this.mongoTemplate.find(query, entity, tableName);
        }

        return ListPageContext.createGenericPager(data, pageSearch.getPageIndex(), pageSearch.getPageSize(), (int)total);
    }

    public MongoTemplate getMongoTemplate() {
        return mongoTemplate;
    }

    public void setMongoTemplate(MongoTemplate mongoTemplate) {
        this.mongoTemplate = mongoTemplate;
    }

    private MongoTemplate mongoTemplate;

    public static final String KEY_ID_DEFAULT = "_id";
}
