package com.walker.jdbc;

//import com.walker.common.SplitPageInfo;
import com.walker.db.Sorts;
import com.walker.db.page.GenericPager;
import com.walker.db.page.MapPager;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;

import java.util.List;
import java.util.Map;

/**
 * 数据库访问DAO接口
 *
 * @author 时克英
 * @date 2017年3月1日
 */
public interface BaseDao {
    /**
     * <pre>
     * 根据sql和条件对象查询PO数组
     * 例如:
     *   List&lt;User&gt; list = selectSplit("select * from user where name like ?", new Object[]{"%admin"}, new SplitPageInfo(6), User.ROW_MAPPER))
     * </pre>
     *
     * @param sql           查询语句
     * @param parameters    参数对象
//     * @param splitPageInfo 分页对象
     * @param rowMapper     resultset转PO对象
     * @param <T>           RowMapper 或 BasePo 子类
     * @return PO数组
     * @throws DataAccessException 数据访问异常
     */
    <T> GenericPager<T> selectSplit(String sql, Object[] parameters, int currentPage, int pageSize, RowMapper<T> rowMapper) throws DataAccessException;

    // 2023-07-27 添加排序参数
    <T> GenericPager<T> selectSplit(String sql, Object[] parameters, int currentPage, int pageSize, RowMapper<T> rowMapper, Sorts.Sort srot) throws DataAccessException;

    /**
     *
     * @param sql
     * @param parameters
     * @param po
     * @param <T>
     * @return
     * @throws DataAccessException
     */
    <T extends BasePo<T>> GenericPager<T> selectSplit(String sql, Object[] parameters, int currentPage, int pageSize, T po) throws DataAccessException;

    /**
     * 插入数据
     *
     * @param po  需要插入的表PO
     * @param <T> BasePo子类
     * @return 成功插入的记录数
     * @throws DataAccessException 数据访问异常
     */
    <T extends BasePo<T>> int insert(T po) throws DataAccessException;

    /**
     * 插入一组PO
     *
     * @param poList 需要插入的表PO数组
     * @param <T>    BasePo子类
     * @return 成功插入的记录数
     * @throws DataAccessException 数据访问异常
     */
    <T extends BasePo<T>> int insert(List<T> poList) throws DataAccessException;

    /**
     * 批量写入数据集合
     * @param poList 给定数据表对象集合。
     * @return 返回成功记录数量，失败返回0
     * @param <T>
     * @date 2023-03-24
     * @author 时克英
     */
    public <T extends BasePo<T>> int insertBatch(List<T> poList) throws DataAccessException;

    /**
     * 根据主键更新数据，部分更新
     *
     * @param po  需要更新的数据库表PO
     * @param <T> BasePo子类
     * @return 成功修改的记录数
     * @throws DataAccessException 数据访问异常
     */
    <T extends BasePo<T>> int update(T po) throws DataAccessException;

    /**
     * 根据各自的主键更新一组数据，部分更新
     *
     * @param poList 需要更新的数据库表PO数组
     * @param <T>    BasePo子类
     * @return 成功修改的记录数
     * @throws DataAccessException 数据访问异常
     */
    <T extends BasePo<T>> int update(List<T> poList) throws DataAccessException;

    /**
     * 批量更新数据集合，使用系统生成的po对象。该方法只根据主键更新
     * @param poList
     * @return
     * @param <T>
     * @throws DataAccessException
     * @author 时克英
     * @date 2023-03-24
     */
    <T extends BasePo<T>> int updateBatch(List<T> poList) throws DataAccessException;

    /**
     * 保存数据, 主键不空则更新，主键为空则插入
     *
     * @param po  需要更新的PO
     * @param <T> BasePo子类
     * @return 影响的数据行数
     */
    <T extends BasePo<T>> int save(T po);

    /**
     * <pre>
     * 根据指定的where条件和parameters参数对象更新数据库
     * 例如：
     *   User user = new User(sybz=1, beiz="null")
     *   int cnt = update(user, "where name like :name and sybz = :sybz", new Map{name : "%admin", sybz : 0})
     * </pre>
     *
     * @param po         需要更新的数据库表PO
     * @param where      条件
     * @param parameters 条件参数
     * @param <T>        BasePo子类
     * @return 成功修改的记录数
     * @throws DataAccessException 数据访问异常
     */
    <T extends BasePo<T>> int update(T po, String where, Map<String, Object> parameters) throws DataAccessException;

    /**
     * <pre>
     * 根据指定的where条件和parameters参数对象更新数据库
     * 例如：
     *   User user = new User(sybz=1, beiz="null")
     *   int cnt = update(user, "where name like ? and sybz = ?", new Object[]{"%admin", 0})
     * </pre>
     *
     * @param po         需要更新的数据库表PO
     * @param where      条件
     * @param parameters 条件参数
     * @param <T>        BasePo子类
     * @return 成功修改的记录数
     * @throws DataAccessException 数据访问异常
     */
    <T extends BasePo<T>> int update(T po, String where, Object[] parameters) throws DataAccessException;

    /**
     * 根据主键删除数据
     *
     * @param po  需要删除的数据库表PO
     * @param <T> BasePo子类
     * @return 成功删除的记录数
     * @throws DataAccessException 数据访问异常
     */
    <T extends BasePo<T>> int delete(T po) throws DataAccessException;

    /**
     * 更新各自的主键删除一组数据
     *
     * @param poList 需要删除的数据库表PO数组
     * @param <T>    BasePo子类
     * @return 成功删除的记录数
     * @throws DataAccessException 数据访问异常
     */
    <T extends BasePo<T>> int delete(List<T> poList) throws DataAccessException;

    /**
     * <pre>
     * 根据条件和条件参数删除数据
     * 例如：
     *   int cnt = delete(new User(), "where sybz = :sybz", new Map{sybz : 3})
     * </pre>
     *
     * @param po         需要删除的数据库表PO
     * @param where      条件
     * @param parameters 条件参数
     * @param <T>        BasePo的子类型
     * @return 成功删除的记录数
     * @throws DataAccessException 数据访问异常
     */
    <T extends BasePo<T>> int delete(T po, String where, Map<String, Object> parameters) throws DataAccessException;

    /**
     * <pre>
     * 根据条件和条件参数删除数据
     * 例如：
     *   int cnt = delete(new User(), "where sybz = ?", new Object[]{3})
     * </pre>
     *
     * @param po         需要删除的数据库表PO
     * @param where      条件
     * @param parameters 条件参数
     * @param <T>        BasePo的子类型
     * @return 成功删除的记录数
     * @throws DataAccessException 数据访问异常
     */
    <T extends BasePo<T>> int delete(T po, String where, Object[] parameters) throws DataAccessException;

    /**
     * 根据主键查询单个PO
     *
     * @param po  数据对象
     * @param <T> BasePo的子类型
     * @return 单个PO对象
     * @throws DataAccessException 数据访问异常
     */
    <T extends BasePo<T>> T get(T po) throws DataAccessException;

    /**
     * <pre>
     * 根据条件和条件对象查询单个PO，如果没查到返回null,查到多余一个报错
     * 例如:
     *   User user = get(new User(), "where name like :name", new Map{name : "%admin"})
     *   User user = get(new User(), "where name = :name", new Map{name : "admin"})
     * </pre>
     *
     * @param po         BasePo的子类型
     * @param where      查询条件
     * @param parameters 条件参数
     * @param <T>        BasePo的子类型
     * @return PO 对象
     * @throws DataAccessException 数据访问异常
     */
    <T extends BasePo<T>> T get(T po, String where, Map<String, Object> parameters) throws DataAccessException;

    /**
     * <pre>
     * 根据条件和条件对象查询单个PO，如果没查到返回null,查到多余一个报错
     * 例如:
     *   User user = get(new User(), "where name like ?", new Object[]{"%admin"})
     *   User user = get(new User(), "where name = ?", new Object[]{"admin"})
     * </pre>
     *
     * @param po         BasePo的子类型
     * @param where      查询条件
     * @param parameters 条件参数
     * @param <T>        BasePo的子类型
     * @return PO 对象
     * @throws DataAccessException 数据访问异常
     */
    <T extends BasePo<T>> T get(T po, String where, Object[] parameters) throws DataAccessException;

    /**
     * <pre>
     * 根据sql和条件对象查询单个Map，如果没查到返回null,查到多余一个报错
     * 例如:
     *   Map<String, Object> userMap = get("select * from user where name like :name", new Map{name : "%admin"})
     *   Map<String, Object> userMap = get("select * from user where name = :name", new Map{name : "admin"})
     * </pre>
     *
     * @param sql        查询语句
     * @param parameters 条件参数
     * @return 数据Map
     * @throws DataAccessException 数据访问异常
     */
    Map<String, Object> get(String sql, Map<String, Object> parameters) throws DataAccessException;

    /**
     * <pre>
     * 根据sql和条件对象查询单个Map，如果没查到返回null,查到多余一个报错
     * 例如:
     *   Map<String, Object> userMap = get("select * from user where name like ?", new Object[]{"%admin"})
     *   Map<String, Object> userMap = get("select * from user where name = ?", new Object[]{"admin"})
     * </pre>
     *
     * @param sql        查询语句
     * @param parameters 条件参数
     * @return Map
     * @throws DataAccessException 数据访问异常
     */
    Map<String, Object> get(String sql, Object[] parameters) throws DataAccessException;

    /**
     * <pre>
     * 根据sql和条件对象查询单个对象，如果没查到返回null,查到多余一个报错
     * 例如:
     *   User user = get("select * from user where name like :name", new Map{name : "%admin"}, User.ROW_MAPPER)
     *   User user = get("select * from user where name = :name", new Map{name : "admin"}, User.ROW_MAPPER)
     * </pre>
     *
     * @param sql        查询语句
     * @param parameters 条件参数
     * @param rowMapper  resultset转PO对象
     * @param <T>        RowMapper 或 BasePo 子类
     * @return T
     * @throws DataAccessException 数据访问异常
     */
    <T> T get(String sql, Map<String, Object> parameters, RowMapper<T> rowMapper) throws DataAccessException;

    /**
     *
     * @param sql
     * @param parameters
     * @param po
     * @param <T>
     * @return
     * @throws DataAccessException
     */
    <T extends BasePo<T>> T get(String sql, Map<String, Object> parameters, T po) throws DataAccessException;

    /**
     * <pre>
     * 根据sql和条件对象查询单个对象，如果没查到返回null,查到多余一个报错
     * 例如:
     *   User user = get("select * from user where name like ?", new Object[]{"%admin"}, User.ROW_MAPPER)
     *   User user = get("select * from user where name = ?", new Object[]{"admin"}, User.ROW_MAPPER)
     * </pre>
     *
     * @param sql        查询语句
     * @param parameters 条件参数
     * @param rowMapper  resultset转PO对象
     * @param <T>        RowMapper 或 BasePo 子类
     * @return T
     * @throws DataAccessException 数据访问异常
     */
    <T> T get(String sql, Object[] parameters, RowMapper<T> rowMapper) throws DataAccessException;

    /**
     *
     * @param sql
     * @param parameters
     * @param po
     * @param <T>
     * @return
     * @throws DataAccessException
     */
    <T extends BasePo<T>> T get(String sql, Object[] parameters, T po) throws DataAccessException;

    /**
     * <pre>
     * 根据条件和条件对象查询PO数组
     * 例如:
     *   List&lt;User&gt; list = select(new User(), "where name like :name", new Map{name : "%admin"})
     * </pre>
     *
     * @param po         需要查询的表PO
     * @param where      条件
     * @param parameters 条件对象
     * @param <T>        BasePo 的子类型
     * @return PO数组
     * @throws DataAccessException 数据访问异常
     */
    <T extends BasePo<T>> List<T> select(T po, String where, Map<String, Object> parameters) throws DataAccessException;

    /**
     * <pre>
     * 根据条件和条件对象查询PO数组
     * 例如:
     *   List&lt;User&gt; list = select(new User(), "where name like ?", new Object[]{"%admin"})
     * </pre>
     *
     * @param po         需要查询的表PO
     * @param where      条件
     * @param parameters 条件对象
     * @param <T>        BasePo 的子类型
     * @return PO数组
     * @throws DataAccessException 数据访问异常
     */
    <T extends BasePo<T>> List<T> select(T po, String where, Object[] parameters) throws DataAccessException;

    /**
     * 通用查询条件，po中可以设置查询字段，目前都是 and 方式。
     * @param po
     * @return
     * @param <T>
     * @throws DataAccessException
     * @date 2023-03-24
     */
    <T extends BasePo<T>> List<T> select(T po) throws DataAccessException;

    // 2023-07-28 sort
    <T extends BasePo<T>> List<T> select(T po, Sorts.Sort sort) throws DataAccessException;

    /**
     * <pre>
     * 根据条件和条件对象查询PO数组
     * 例如:
     *   List&lt;User&gt; list = selectSplit(new User(), "where name like :name", new Map{name : "%admin"}, new SplitPageInfo(6))
     * </pre>
     *
     * @param po            需要查询的表PO
     * @param where         条件
     * @param parameters    条件对象
     * @param <T>           BasePo 子类型
     * @return 分页后的PO数组
     * @throws DataAccessException 数据访问异常
     */
    <T extends BasePo<T>> GenericPager<T> selectSplit(T po
            , String where, Map<String, Object> parameters, int currentPage, int pageSize) throws DataAccessException;

    /**
     * 简单的单表分页查询，设置PO字段可查询，目前只支持 and 条件。
     * @param po
     * @param currentPage
     * @param pageSize
     * @return
     * @param <T>
     * @throws DataAccessException
     * @date 2023-03-24
     */
    public <T extends BasePo<T>> GenericPager<T> selectSplit(T po, int currentPage, int pageSize) throws DataAccessException;

    // 2023-07-28 sort
    public <T extends BasePo<T>> GenericPager<T> selectSplit(T po, int currentPage, int pageSize, Sorts.Sort sort) throws DataAccessException;

    /**
     * <pre>
     * 根据sql和条件对象查询map数组
     * 例如:
     *   List&lt;Map&lt;String, Object&gt;&gt; list = select("select * from user where name like ?", new Object[]{"%admin"})
     * </pre>
     *
     * @param sql        查询语句
     * @param parameters 条件对象
     * @return PO数组
     * @throws DataAccessException 数据访问异常
     */
    List<Map<String, Object>> select(String sql, Object[] parameters) throws DataAccessException;

    /**
     * <pre>
     * 根据sql和条件对象查询PO数组
     * 例如:
     *   List&lt;User&gt; list = select("select * from user where name like ?", new Object[]{"%admin"}, User.ROW_MAPPER)
     * </pre>
     *
     * @param sql        查询语句
     * @param parameters 条件对象
     * @param mapper     resultset 转换对象
     * @param <T>        RowMapper 或 BasePo 子类型
     * @return PO数组
     * @throws DataAccessException 数据访问异常
     */
    <T> List<T> select(String sql, Object[] parameters, RowMapper<T> mapper) throws DataAccessException;

    /**
     *
     * @param sql
     * @param parameters
     * @param po
     * @param <T>
     * @return
     * @throws DataAccessException
     */
    <T extends BasePo<T>> List<T> select(String sql, Object[] parameters, T po) throws DataAccessException;

    /**
     * <pre>
     * 根据sql和条件对象查询map数组
     * 例如:
     *   List&lt;Map&lt;String, Object&gt;&gt; list = selectSplit("select * from user where name like ?", new Object[]{"%admin"}, new SplitPageInfo(6))
     * </pre>
     *
     * @param sql           查询语句
     * @param parameters    条件对象
//     * @param splitPageInfo 分页对象
     * @param currentPage 当前页码
     * @param pageSize 每页显示几条
     * @return PO数组
     * @throws DataAccessException 数据访问异常
     */
    MapPager selectSplit(String sql, Object[] parameters, int currentPage, int pageSize) throws DataAccessException;

    // 2023-07-28 添加Sort排序参数
    MapPager selectSplit(String sql, Object[] parameters, int currentPage, int pageSize, Sorts.Sort sort) throws DataAccessException;

    /**
     * <pre>
     * 根据sql和条件对象查询map数组，结果放在分页对象中
     * 例如:
     *   SplitPageInfo spi = selectSplitEx("select * from user where name like ?", new Object[]{"%admin"}, new SplitPageInfo(6))
     * </pre>
     *
     * @param sql           查询语句
     * @param parameters    条件对象
     * @param splitPageInfo 分页对象
     * @return 分页对象
     * @throws DataAccessException 数据访问异常
    SplitPageInfo selectSplitEx(String sql, Object[] parameters, SplitPageInfo splitPageInfo) throws DataAccessException;*/

//    /**
//     * <pre>
//     * 根据sql和条件对象查询PO数组，结果放在分页对象中返回
//     * 例如:
//     *   SplitPageInfo spi = selectSplitEx("select * from user where name like ?", new Object[]{"%admin"}, new SplitPageInfo(6), User.ROW_MAPPER)
//     * </pre>
//     *
//     * @param sql           查询语句
//     * @param parameters    参数对象
//     * @param splitPageInfo 分页对象
//     * @param rowMapper     resultset转PO对象
//     * @param <T>           RowMapper 或 BasePo 子类
//     * @return 分页对象
//     * @throws DataAccessException 数据访问异常
//     */
//    <T> SplitPageInfo selectSplitEx(String sql, Object[] parameters, SplitPageInfo splitPageInfo, RowMapper<T> rowMapper) throws DataAccessException;

//    /**
//     *
//     * @param sql
//     * @param parameters
//     * @param splitPageInfo
//     * @param po
//     * @param <T>
//     * @return
//     * @throws DataAccessException
//     */
//    <T extends BasePo<T>> SplitPageInfo selectSplitEx(String sql, Object[] parameters, SplitPageInfo splitPageInfo, T po) throws DataAccessException;

    /**
     * <pre>
     * 根据sql和条件对象分页查询map数组，结果放在分页对象中返回
     * 例如:
     *   List&lt;User&gt; list = selectSplit("select * from user where name like :name", new Map{name:"%admin"}, new SplitPageInfo(6), User.ROW_MAPPER)
     * </pre>
     *
     * @param sql           查询语句
     * @param parameters    条件参数
//     * @param splitPageInfo 分页对象
     * @param currentPage 当前页码
     * @param pageSize 每页显示几条
     * @param rowMapper     resultset转PO对象
     * @param <T>           RowMapper 或 BasePo 子类
     * @return PO数组
     * @throws DataAccessException 数据访问异常
     */
    <T> GenericPager<T> selectSplit(String sql, Map<String, Object> parameters, int currentPage, int pageSize, RowMapper<T> rowMapper) throws DataAccessException;
    <T> GenericPager<T> selectSplit(String sql, Map<String, Object> parameters, int currentPage, int pageSize, RowMapper<T> rowMapper, Sorts.Sort sort) throws DataAccessException;

    <T> GenericPager<T> selectSplit(String sql, Map<String, Object> parameters, RowMapper<T> rowMapper, Sorts.Sort sort) throws DataAccessException;

    <T extends BasePo<T>> GenericPager<T> selectSplit(String sql, Map<String, Object> parameters, int currentPage, int pageSize, T po) throws DataAccessException;

    /**
     * <pre>
     * 根据sql和条件对象查询map数组
     * 例如:
     *   List&lt;Map&lt;String, Object&gt;&gt; list = select("select * from user where name like ?", new Map{"%admin"})
     * </pre>
     *
     * @param sql        查询语句
     * @param parameters 参数对象
     * @return map数组
     * @throws DataAccessException 数据访问异常
     */
    List<Map<String, Object>> select(String sql, Map<String, Object> parameters) throws DataAccessException;

    /**
     * <pre>
     * 根据sql和条件对象查询map数组
     * 例如:
     *   List&lt;User&gt; list = select("select * from user where name like ?", new Object[]{"%admin"}, User.ROW_MAPPER)
     * </pre>
     *
     * @param sql        查询语句
     * @param parameters 参数对象
     * @param mapper     resultset转PO对象
     * @param <T>        RowMapper 或 BasePo 子类
     * @return T数组
     * @throws DataAccessException 数据访问异常
     */
    <T> List<T> select(String sql, Map<String, Object> parameters, RowMapper<T> mapper) throws DataAccessException;

    /**
     *
     * @param sql
     * @param parameters
     * @param po
     * @param <T>
     * @return
     * @throws DataAccessException
     */
    <T extends BasePo<T>> List<T> select(String sql, Map<String, Object> parameters, T po) throws DataAccessException;

    /**
     * <pre>
     * 根据sql和条件对象分页查询map数组
     * 例如:
     *   List&lt;Map&lt;String, Object&gt;&gt; list = selectSplit("select * from user where name like :name", new Map{name:"%admin"}, new SplitPageInfo(6))
     * </pre>
     *
     * @param sql           查询语句
     * @param parameters    参数对象
//     * @param splitPageInfo 分页对象
     * @return map数组
     * @throws DataAccessException 数据访问异常
     */
    MapPager selectSplit(String sql, Map<String, Object> parameters, int currentPage, int pageSize) throws DataAccessException;

    // 2023-07-27 Sort
    MapPager selectSplit(String sql, Map<String, Object> parameters, int currentPage, int pageSize, Sorts.Sort sort) throws DataAccessException;

//    /**
//     * <pre>
//     * 根据sql和条件对象分页查询map数组，结果放在分页对象中返回
//     * 例如:
//     *   SplitPageInfo spi = selectSplitEx("select * from user where name like :name", new Map{name:"%admin"}, new SplitPageInfo(6))
//     * </pre>
//     *
//     * @param sql           查询语句
//     * @param parameters    条件参数
//     * @param splitPageInfo 分页对象
//     * @return 分页对象
//     * @throws DataAccessException 数据访问异常
//     */
//    SplitPageInfo selectSplitEx(String sql, Map<String, Object> parameters, SplitPageInfo splitPageInfo) throws DataAccessException;

//    /**
//     * <pre>
//     * 根据sql和条件对象分页查询map数组，结果放在分页对象中返回
//     * 例如:
//     *   SplitPageInfo spi = selectSplitEx("select * from user where name like :name", new Map{name:"%admin"}, new SplitPageInfo(6), User.ROW_MAPPER)
//     * </pre>
//     *
//     * @param sql           查询语句
//     * @param parameters    条件参数
//     * @param splitPageInfo 分页对象
//     * @param rowMapper     resultset转PO对象
//     * @param <T>           RowMapper 或 BasePo 子类
//     * @return 分页对象
//     * @throws DataAccessException 数据访问异常
//     */
//    <T> SplitPageInfo selectSplitEx(String sql, Map<String, Object> parameters, SplitPageInfo splitPageInfo, RowMapper<T> rowMapper) throws DataAccessException;

//    /**
//     *
//     * @param sql
//     * @param parameters
//     * @param splitPageInfo
//     * @param po
//     * @param <T>
//     * @return
//     * @throws DataAccessException
//     */
//    <T extends BasePo<T>> SplitPageInfo selectSplitEx(String sql, Map<String, Object> parameters, SplitPageInfo splitPageInfo, T po) throws DataAccessException;

    /**
     * 处理大结果集查询（不内存溢出）
     *
     * @param sql          SQL 查询语句
     * @param parameters   查询参数
     * @param rowMapper    resultset转PO对象
     * @param rowExecution 执行行业务逻辑对象
     * @param <T>          泛型类型
     */
    <T> void execute(String sql, final Object[] parameters, final RowMapper<T> rowMapper, final RowExecution<T> rowExecution);

    /**
     *
     * @param sql
     * @param parameters
     * @param po
     * @param rowExecution
     * @param <T>
     */
    <T extends BasePo<T>> void execute(String sql, final Object[] parameters, final T po, final RowExecution<T> rowExecution);

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    //~ 以下方法简化分页参数
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    <T extends BasePo<T>> GenericPager<T> selectSplit(String sql
            , Map<String, Object> parameters, T po) throws DataAccessException;

    <T extends BasePo<T>> GenericPager<T> selectSplit(String sql
            , Object[] parameters, T po) throws DataAccessException;

    <T> GenericPager<T> selectSplit(String sql, Object[] parameters, RowMapper<T> rowMapper) throws DataAccessException;

    MapPager selectSplit(String sql, Object[] parameters) throws DataAccessException;

    <T extends BasePo<T>> GenericPager<T> selectSplit(T po
            , String where, Map<String, Object> parameters) throws DataAccessException;

    /**
     * <pre>
     * 根据条件参数更新数据库（insert or udpate or delete）
     * 例如:
     *   int cnt = execute("update user set name=? where id = ?", new Object[]{"张三", "123"})
     * </pre>
     *
     * @param sql        更新语句
     * @param parameters 条件参数
     * @return 受影响的数据行数
     * @throws DataAccessException 数据访问异常
     */
    int execute(String sql, Object[] parameters) throws DataAccessException;

    /**
     * <pre>
     * 根据条件参数更新数据库（insert or udpate or delete）
     * 例如:
     *   int cnt = execute("update user set name=:name where id=:id", new Map{name:"张三", id:"123"})
     *  </pre>
     *
     * @param sql        更新语句
     * @param parameters 条件参数
     * @return 受影响的数据行数
     * @throws DataAccessException 数据访问异常
     */
    int execute(String sql, Map<String, Object> parameters) throws DataAccessException;

    /**
     * <pre>
     * 根据sql语句和条件参数查询一个Int
     * 例如:
     *   int count = queryForInt("select count(*) from user where sybz=?", new Object[]{1})
     * </pre>
     *
     * @param sql        查询语句
     * @param parameters 参数对象
     * @return int
     * @throws DataAccessException 数据访问异常
     */
    int queryForInt(String sql, Object[] parameters) throws DataAccessException;

    /**
     * <pre>
     * 根据sql语句和条件参数查询一个Int
     * 例如:
     *   int count = queryForInt("select count(*) from user where sybz=:sybz", new Map{sybz:1})
     * </pre>
     *
     * @param sql        查询语句
     * @param parameters 参数对象
     * @return int
     * @throws DataAccessException 数据访问异常
     */
    int queryForInt(String sql, Map<String, Object> parameters) throws DataAccessException;


    /**
     * <pre>
     * 根据sql语句和条件参数查询一个Object
     * 例如:
     *   String name = queryForObject("select name from user where id=?", new Object[]{"123"}, String.class)
     * </pre>
     *
     * @param sql        查询语句
     * @param parameters 参数对象
     * @param clazz      返回的数据类型
     * @param <T>        对象类型
     * @return T类型的对象
     * @throws DataAccessException 数据访问异常
     */
    <T> T queryForObject(String sql, Object[] parameters, Class<T> clazz) throws DataAccessException;

    /**
     * <pre>
     * 根据sql语句和条件参数查询一个Object
     * 例如:
     *   String name = queryForObject("select name from user where id=:id", new Map[]{id:"123"}, String.class)
     * </pre>
     *
     * @param sql        查询语句
     * @param parameters 参数对象
     * @param clazz      返回的数据类型
     * @param <T>        对象类型（例如，String/Integer等）
     * @return T类型的对象
     * @throws DataAccessException 数据访问异常
     */
    <T> T queryForObject(String sql, Map<String, Object> parameters, Class<T> clazz) throws DataAccessException;

    /**
     * <pre>
     * 执行存储过程
     * 例如
     *   call("{call bbb(?, ?)}", new Integer[]{1, 2});
     * </pre>
     *
     * @param functionName 方法名称
     * @param parameters   参数对象
     * @throws DataAccessException 数据访问异常
     */
    void execCall(String functionName, Object[] parameters) throws DataAccessException;

    /**
     * <pre>
     * 执行存储过程
     * 例如
     *   Integer cnt = call("{? = call add(?, ?)}", new Integer[]{1, 2}, Integer.class);
     * </pre>
     *
     * @param functionName 方法名称
     * @param parameters   参数对象
     * @param clazz        返回对象类型
     * @param <T>          泛型类型
     * @return 执行结果
     * @throws DataAccessException 数据访问异常
     */
    <T> T execCall(String functionName, Object[] parameters, Class<T> clazz) throws DataAccessException;

    /**
     * <pre>
     * 批量更新
     * 例如：
     *   params = new List[
     *     new Object[]{1, "123"},
     *     new Object[]{1, "124"},
     *     new Object[]{1, "125"}
     *   ]
     *   int cnt = batchUpdate("update user sybz=? where id=?", params);
     *
     *   params = new List[
     *     new Map{sybz:1, id:"123"},
     *     new Map{sybz:1, id:"124"},
     *     new Map{sybz:1, id:"125"}
     *   ]
     *   int cnt = batchUpdate("update user sybz=:sybz where id=:id", params);
     * </pre>
     *
     * @param sql            更新语句
     * @param parametersList 参数对象数组
     * @return 受影响的行数
     * @throws DataAccessException 数据访问异常
     */
    int execBatchUpdate(String sql, List<?> parametersList) throws DataAccessException;

    void update(String sql);

    int update(String sql, Object[] args);

    <T> List<T> sqlQuery(String sql, RowMapper<T> rowMapper);

    <T> List<T> sqlQuery(String sql, Object[] args, RowMapper<T> rowMapper);

    List<Map<String, Object>> sqlQueryListMap(String sql, Object[] args);

    <E> GenericPager<E> sqlGeneralQueryPager(String sql, Object[] args, RowMapper<E> rowMapper);

    <E> GenericPager<E> sqlGeneralQueryPager(String sql, Object[] args, RowMapper<E> rowMapper
            , int pageIndex);

    <T> GenericPager<T> sqlGeneralQueryPager(String sql
            , Object[] args, RowMapper<T> rowMapper, int pageIndex, int pageSize);

    <T> T sqlMathQuery(String sql, Object[] args, Class<T> clazz);

    <T> List<T> sqlListObjectWhereIn(String sql, RowMapper<T> rowMapper, SqlParameterSource paramSource);

    List<Map<String, Object>> queryListObjectWhereIn(String sql, SqlParameterSource paramSource);

    /**
     * 分页查询，根据PO里面字段，做简单条件查询。
     * @param po
     * @return
     * @param <T>
     * @throws DataAccessException
     * @date 2023-04-21
     */
    <T extends BasePo<T>> GenericPager<T> selectSplit(T po) throws DataAccessException;
    <T extends BasePo<T>> GenericPager<T> selectSplit(T po, Sorts.Sort sort) throws DataAccessException;

    static interface RowExecution<T> {
        void execute(T t) throws Exception;
    }
}
