package com.walker.jdbc.dao;

//import com.walker.common.SplitPageInfo;

import com.walker.db.Sorts;
import com.walker.db.page.GenericPager;
import com.walker.db.page.ListPageContext;
import com.walker.db.page.MapPager;
import com.walker.db.page.PageSearch;
import com.walker.infrastructure.utils.NumberGenerator;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.jdbc.BaseDao;
import com.walker.jdbc.BaseMapper;
import com.walker.jdbc.BasePo;
import com.walker.jdbc.ColumnMapRowMapper;
import com.walker.jdbc.SqlAndParameters;
import com.walker.jdbc.util.JdbcUtils;
import com.walker.jdbc.util.StringSqlUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.CallableStatementCallback;
import org.springframework.jdbc.core.CallableStatementCreator;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.PreparedStatementCallback;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.SingleColumnRowMapper;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;
import org.springframework.util.Assert;

import java.math.BigDecimal;
import java.sql.Blob;
import java.sql.CallableStatement;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 数据库访问jdbc DAO实现
 *
 * @author 时克英
 * @date 2017年3月1日
 */
public class JdbcDaoImpl implements BaseDao {
    protected final transient Logger logger = LoggerFactory.getLogger(JdbcDaoImpl.class);
    private static final Pattern ARRAY_PARAMETER_PLACEHOLDER = Pattern.compile("\\?");
    private JdbcTemplate jdbcTemplate;
    private NamedParameterJdbcTemplate namedParameterJdbcTemplate;
    private PaginationHelper paginationHelper;
    private boolean showSql = true;

    public PaginationHelper getPaginationHelper() {
        return paginationHelper;
    }

    public void setPaginationHelper(PaginationHelper paginationHelper) {
        this.paginationHelper = paginationHelper;
    }

    public JdbcTemplate getJdbcTemplate() {
        return jdbcTemplate;
    }

    public void setJdbcTemplate(JdbcTemplate jdbcTemplate) {
        this.jdbcTemplate = jdbcTemplate;
    }

    public NamedParameterJdbcTemplate getNamedParameterJdbcTemplate() {
        return namedParameterJdbcTemplate;
    }

    public void setNamedParameterJdbcTemplate(NamedParameterJdbcTemplate namedParameterJdbcTemplate) {
        this.namedParameterJdbcTemplate = namedParameterJdbcTemplate;
    }

    public boolean isShowSql() {
        return showSql;
    }

    public void setShowSql(boolean showSql) {
        this.showSql = showSql;
    }

    private <T extends BasePo<T>> BaseMapper<T> convertToMapper(T basePo) {
        return JdbcUtils.getInstance(basePo);
    }

    /**
     * @see com.walker.jdbc.BaseDao#insert(T)
     */
    @Override
    public <T extends BasePo<T>> int insert(T po) throws DataAccessException {
        Assert.notNull(po, "po 不能为空！");
        BaseMapper pomapper = this.convertToMapper(po);
        Assert.notNull(pomapper.getPkValue_(), po.getClass().getCanonicalName() + "主键 " + pomapper.getPkName_() + " 不能为空");

        SqlAndParameters<Map<String, Object>> sp = pomapper.getInsertSql_();
        printSql(sp);
        return this.namedParameterJdbcTemplate.update(sp.getSql(), sp.getParameters());
    }

    /**
     * 批量写入数据集合
     * @param poList 给定数据表对象集合。
     * @return 返回成功记录数量，失败返回0
     * @param <T>
     * @date 2023-03-24
     * @author 时克英
     */
    public <T extends BasePo<T>> int insertBatch(List<T> poList) throws DataAccessException{
        if(poList == null || poList.size() == 0){
            throw new IllegalArgumentException("poList is required!");
        }
        String sql = null;
        Map<String, Object>[] parameters = new HashMap[poList.size()];
        BasePo basePo = null;
        BaseMapper baseMapper = null;
        SqlAndParameters<Map<String, Object>> sp = null;
        for(int i=0; i< poList.size(); i++){
            basePo = poList.get(i);
            baseMapper = this.convertToMapper(basePo);
            sp = baseMapper.getInsertSql_();
            if(sql == null){
                sql = sp.getSql();
            }
            parameters[i] = sp.getParameters();
        }
        int[] result = this.namedParameterJdbcTemplate.batchUpdate(sql, parameters);
        if(result != null){
            return result[0];
        }
        return 0;
    }

    /**
     * 打印SQL
     *
     * @param sp SQL和参数对象
     */
    private void printSql(SqlAndParameters sp) {
        if (this.isShowSql()) {
            logger.info("{}", sp.toPrintSql());
        }
    }

    /**
     * @see com.walker.jdbc.BaseDao#insert(List<T> )
     */
    @Override
    public <T extends BasePo<T>> int insert(List<T> poList) throws DataAccessException {
        Assert.notNull(poList, "poList 不能为空！");
        int sum = 0;
        for (BasePo po : poList) {
            sum += this.insert(po);
        }
        return sum;
    }

    /**
     * @see com.walker.jdbc.BaseDao#update(T)
     */
    @Override
    public <T extends BasePo<T>> int update(T po) throws DataAccessException {
        Assert.notNull(po, "po 不能为空！");
        BaseMapper pomapper = this.convertToMapper(po);
        Assert.notNull(pomapper.getPkValue_(), po.getClass().getCanonicalName() + "主键 " + pomapper.getPkName_() + " 不能为空");

        SqlAndParameters<Map<String, Object>> sp = pomapper.getUpdateSql_();
        printSql(sp);
        return this.namedParameterJdbcTemplate.update(sp.getSql(), sp.getParameters());
    }

    /**
     * @see BaseDao#save(T)
     */
    @Override
    public <T extends BasePo<T>> int save(T po) {
        BaseMapper pomapper = this.convertToMapper(po);
        SqlAndParameters<Map<String, Object>> sp = null;

        if (pomapper.getPkValue_() == null || StringUtils.EMPTY_STRING.equals(pomapper.getPkValue_())) {
            // 2022-10-11 如果主键为空自动生成一个long(纳秒值)
//            pomapper.setPkValue(NumberGenerator.getLongSequenceId());
            pomapper.setPkValue(NumberGenerator.getLongSequenceNumber());
            po.setPkValue(pomapper.getPkValue_());
            sp = pomapper.getInsertSql_();
        } else {
            sp = pomapper.getUpdateSql_();
        }
        printSql(sp);
        return this.namedParameterJdbcTemplate.update(sp.getSql(), sp.getParameters());
    }

    /**
     * @see com.walker.jdbc.BaseDao#update(List<T> )
     */
    @Override
    public <T extends BasePo<T>> int update(List<T> poList) throws DataAccessException {
        Assert.notNull(poList, "poList 不能为空！");
        int sum = 0;
        for (BasePo po : poList) {
            sum += this.update(po);
        }
        return sum;
    }

    /**
     * 批量更新数据集合，使用系统生成的po对象。该方法只根据主键更新
     * @param poList
     * @return
     * @param <T>
     * @throws DataAccessException
     * @author 时克英
     * @date 2023-03-24
     */
    public <T extends BasePo<T>> int updateBatch(List<T> poList) throws DataAccessException{
        if(poList == null || poList.size() == 0){
            return 0;
        }
        String sql = null;
        Map<String, Object>[] parameters = new HashMap[poList.size()];
        BasePo basePo = null;
        BaseMapper baseMapper = null;
        SqlAndParameters<Map<String, Object>> sp = null;
        for(int i=0; i< poList.size(); i++){
            basePo = poList.get(i);
            baseMapper = this.convertToMapper(basePo);
            sp = baseMapper.getUpdateSql_();
            if(sql == null){
                sql = sp.getSql();
            }
            parameters[i] = sp.getParameters();
        }
        int[] result = this.namedParameterJdbcTemplate.batchUpdate(sql, parameters);
        if(result != null){
            return result[0];
        }
        return 0;
    }

    /**
     * @see com.walker.jdbc.BaseDao#update(T, String, Map< String, Object> )
     */
    @Override
    public <T extends BasePo<T>> int update(T po, String where, Map<String, Object> parameters) throws DataAccessException {
        Assert.notNull(po, "po 不能为空!");
        Assert.notNull(where, "where 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");
        BaseMapper pomapper = this.convertToMapper(po);
        SqlAndParameters<Map<String, Object>> sp = pomapper.getUpdateSql_(where, parameters);
        printSql(sp);
        return this.namedParameterJdbcTemplate.update(sp.getSql(), sp.getParameters());
    }

    @Override
    public <T extends BasePo<T>> int update(T po, String where, Object[] parameters) throws DataAccessException {
        Assert.notNull(po, "po 不能为空!");
        Assert.notNull(where, "where 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");
        BaseMapper pomapper = this.convertToMapper(po);
        SqlAndParameters<Object[]> sp = pomapper.getUpdateSql_(where, parameters);
        printSql(sp);
        return this.jdbcTemplate.update(sp.getSql(), sp.getParameters());
    }

    /**
     * @see com.walker.jdbc.BaseDao#delete(T)
     */
    @Override
    public <T extends BasePo<T>> int delete(T po) throws DataAccessException {
        Assert.notNull(po, "po 不能为空！");
        BaseMapper pomapper = this.convertToMapper(po);
        Assert.notNull(pomapper.getPkValue_(), po.getClass().getCanonicalName() + " 主键 " + pomapper.getPkName_() + " 不能为空");

        SqlAndParameters<Map<String, Object>> sp = pomapper.getDeleteSql_();
        printSql(sp);
        return this.namedParameterJdbcTemplate.update(sp.getSql(), sp.getParameters());
    }

    /**
     * @see com.walker.jdbc.BaseDao#delete(List<T> )
     */
    @Override
    public <T extends BasePo<T>> int delete(List<T> poList) throws DataAccessException {
        Assert.notNull(poList, "poList 不能为空！");
        int sum = 0;
        for (BasePo po : poList) {
            sum += this.delete(po);
        }
        return sum;
    }

    /**
     * @see com.walker.jdbc.BaseDao#delete(T, String, Map< String, Object> )
     */
    @Override
    public <T extends BasePo<T>> int delete(T po, String where, Map<String, Object> parameters) throws DataAccessException {
        Assert.notNull(po, "po 不能为空！");
        Assert.notNull(where, "where 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");
        BaseMapper pomapper = this.convertToMapper(po);
        SqlAndParameters<Map<String, Object>> sp = pomapper.getDeleteSql_(where, parameters);
        printSql(sp);
        return namedParameterJdbcTemplate.update(sp.getSql(), sp.getParameters());
    }

    @Override
    public <T extends BasePo<T>> int delete(T po, String where, Object[] parameters) throws DataAccessException {
        Assert.notNull(po, "po 不能为空！");
        Assert.notNull(where, "where 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");
        BaseMapper pomapper = this.convertToMapper(po);
        SqlAndParameters<Object[]> sp = pomapper.getDeleteSql_(where, parameters);
        printSql(sp);
        return jdbcTemplate.update(sp.getSql(), sp.getParameters());
    }

    /**
     * @see com.walker.jdbc.BaseDao#get(T)
     */
    @Override
    public <T extends BasePo<T>> T get(T po) throws DataAccessException {
        Assert.notNull(po, "po 不能为空！");
        BaseMapper pomapper = this.convertToMapper(po);
        Assert.notNull(pomapper.getPkValue_(), po.getClass().getCanonicalName() + "主键 " + pomapper.getPkName_() + " 不能为空");

        SqlAndParameters<Map<String, Object>> sp = pomapper.getSingleSql_();
        List<T> list = this.select(sp.getSql(), sp.getParameters(), po);
        return getUnique(list);
    }

    /**
     * @see com.walker.jdbc.BaseDao#get(T, String, Map< String, Object> )
     */
    @Override
    public <T extends BasePo<T>> T get(T po, String where, Map<String, Object> parameters) throws DataAccessException {
        Assert.notNull(po, "po 不能为空！");
        Assert.notNull(where, "where 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");
        BaseMapper pomapper = this.convertToMapper(po);
        SqlAndParameters<Map<String, Object>> sp = pomapper.getSelectSql_(where, parameters);
        List<T> list = this.select(sp.getSql(), sp.getParameters(), po);
        return getUnique(list);
    }

    @Override
    public <T extends BasePo<T>> T get(T po, String where, Object[] parameters) throws DataAccessException {
        Assert.notNull(po, "po 不能为空！");
        Assert.notNull(where, "where 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");
        BaseMapper pomapper = this.convertToMapper(po);
        SqlAndParameters<Object[]> sp = pomapper.getSelectSql_(where, parameters);
        List<T> list = this.select(sp.getSql(), sp.getParameters(), po);
        return getUnique(list);
    }

    @Override
    public <T> T get(String sql, Map<String, Object> parameters, RowMapper<T> rowMapper) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空！");
        Assert.notNull(parameters, "parameters 不能为空!");
        Assert.notNull(rowMapper, "rowMapper 不能为空!");

        List<T> list = this.select(sql, parameters, rowMapper);
        return getUnique(list);
    }

    @Override
    public <T extends BasePo<T>> T get(String sql, Map<String, Object> parameters, T po) throws DataAccessException {
        return this.get(sql, parameters, this.convertToMapper(po));
    }

    @Override
    public <T> T get(String sql, Object[] parameters, RowMapper<T> rowMapper) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空！");
        Assert.notNull(parameters, "parameters 不能为空!");
        Assert.notNull(rowMapper, "rowMapper 不能为空!");

        List<T> list = this.select(sql, parameters, rowMapper);
        return getUnique(list);
    }

    @Override
    public <T extends BasePo<T>> T get(String sql, Object[] parameters, T po) throws DataAccessException {
        return this.get(sql, parameters, this.convertToMapper(po));
    }

    /**
     * @see com.walker.jdbc.BaseDao#get(String, Map< String, Object> )
     */
    @Override
    public Map<String, Object> get(String sql, Map<String, Object> parameters) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");

        List<Map<String, Object>> list = this.select(sql, parameters);
        return getUnique(list);
    }

    /**
     * @see com.walker.jdbc.BaseDao#get(String, Object[])
     */
    @Override
    public Map<String, Object> get(String sql, Object[] parameters) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");

        List<Map<String, Object>> list = this.select(sql, parameters);
        return getUnique(list);
    }

    /**
     * @see com.walker.jdbc.BaseDao#select(T, String, Map< String, Object> )
     */
    @Override
    public <T extends BasePo<T>> List<T> select(T po, String where, Map<String, Object> parameters) throws DataAccessException {
        Assert.notNull(po, "po 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");
        BaseMapper pomapper = this.convertToMapper(po);
        SqlAndParameters<Map<String, Object>> sp = pomapper.getSelectSql_(where, parameters);
        return this.select(sp.getSql(), sp.getParameters(), po);
    }

    /**
     * @see com.walker.jdbc.BaseDao#select(T, String, Object[])
     */
    @Override
    public <T extends BasePo<T>> List<T> select(T po, String where, Object[] parameters) throws DataAccessException {
        Assert.notNull(po, "po 不能为空!");
//        Assert.notNull(parameters, "parameters 不能为空!");
        if(parameters == null){
            parameters = new Object[]{};
        }
        BaseMapper pomapper = this.convertToMapper(po);
        SqlAndParameters<Object[]> sp = pomapper.getSelectSql_(where, parameters);
        return this.select(sp.getSql(), sp.getParameters(), po);
    }

    @Override
    public <T extends BasePo<T>> List<T> select(T po) throws DataAccessException {
        return this.select(po, null);
    }

    @Override
    public <T extends BasePo<T>> List<T> select(T po, Sorts.Sort sort) throws DataAccessException{
        if(po == null){
            throw new IllegalArgumentException("po is required!");
        }
//        // po转成json对象，获取里面所有存在值的字段信息
//        ObjectNode objectNode = JsonUtils.javaObjectToObjectNode(po);
//
//        Map.Entry<String, JsonNode> entry = null;
//        String fieldName = null;
//        String valueText = null;
//        JsonNode jsonNode = null;
//        // 字段名、字段值集合
//        Map<String, Object> fieldNameAndValue = new HashMap<>(4);
//        StringBuilder where = new StringBuilder(Constants.SQL_NAME_WHERE);
//        int i = 0;
//        for(Iterator<Map.Entry<String, JsonNode>> it = objectNode.fields(); it.hasNext();){
//            entry = it.next();
//            if(entry.getKey().equals(Constants.PO_FIELD_PARAMETER)){
//                // 额外的参数不属于表字段，剔除
//                continue;
//            }
//
//            jsonNode = entry.getValue();
//
//            // 2023-03-24 注意：这里jsonNode可能为 "null" 字符串
//            if(jsonNode == null || jsonNode.toString().equals(StringUtils.NULL_STRING)){
////                logger.debug("........entry.getValue() = null");
//                continue;
//            }
//
//            // 字段名字，转成下划线数据库字段名
//            fieldName = StringUtils.transferCamelName2Underline(entry.getKey());
//
////            logger.debug("{}", jsonNode);
//            if(i > 0){
//                where.append(StringUtils.CHAR_SPACE).append(Constants.SQL_NAME_AND);
//            }
//
//            if(jsonNode.isTextual()){
//                valueText = jsonNode.asText();
//                logger.debug("valueText = {}", valueText);
//                if(StringUtils.isEmpty(valueText)){
//                    valueText = StringUtils.EMPTY_STRING;
//                }
//                if(valueText.indexOf(StringUtils.CHAR_PERCENT) >= 0){
//                    // like 查询
//                    where.append(StringUtils.CHAR_SPACE).append(fieldName).append(StringUtils.CHAR_SPACE)
//                            .append(Constants.SQL_NAME_LIKE).append(StringUtils.CHAR_SPACE).append(StringUtils.SEPARATOR_COLON).append(fieldName);
//                } else {
//                    // 普通字符串查询
//                    where.append(StringUtils.CHAR_SPACE).append(fieldName)
//                            .append(StringUtils.CHAR_EQUALS).append(StringUtils.SEPARATOR_COLON).append(fieldName);
//                }
//                // 字符串会带双引号，奇怪！2023-03-24
//                valueText = valueText.replace(StringUtils.STRING_DOUBLE_MARK, StringUtils.EMPTY_STRING);
//                fieldNameAndValue.put(fieldName, valueText);
//
//            } else {
//                where.append(StringUtils.CHAR_SPACE).append(fieldName)
//                        .append(StringUtils.CHAR_EQUALS).append(StringUtils.SEPARATOR_COLON).append(fieldName);
//            }
//
//            if(jsonNode.isLong()){
//                fieldNameAndValue.put(fieldName, jsonNode.longValue());
//            } else if(jsonNode.isDouble()){
//                fieldNameAndValue.put(fieldName, jsonNode.doubleValue());
//            } else if(jsonNode.isInt()){
//                fieldNameAndValue.put(fieldName, jsonNode.intValue());
//            } else if(jsonNode.isFloat()){
//                fieldNameAndValue.put(fieldName, jsonNode.floatValue());
//            } else {
//                fieldNameAndValue.put(fieldName, jsonNode.asText());
//            }
//            i++;
//        }

        Object[] result = JdbcUtils.acquireQueryCondition(po);

        BaseMapper pomapper = this.convertToMapper(po);
        SqlAndParameters<Map<String, Object>> sp = pomapper.getSelectSql_(result[0].toString(), (Map<String, Object>)result[1]);
        if(logger.isDebugEnabled()){
            logger.debug(sp.getSql());
            logger.debug("fieldNameAndValue={}", result[1]);
        }
        String sql = sort == null? sp.getSql() : this.paginationHelper.getOrderBySql(sp.getSql(), sort);
        return this.select(sql, sp.getParameters(), po);
//        return this.select(sp.getSql(), sp.getParameters(), po);
    }

    /**
     * @see BaseDao#selectSplit(BasePo, String, Map, int, int)
     */
    @Override
    public <T extends BasePo<T>> GenericPager<T> selectSplit(T po
            , String where, Map<String, Object> parameters, int currentPage, int pageSize) throws DataAccessException {
        Assert.notNull(po, "po 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");
        if(parameters == null){
            parameters = new HashMap<>(2);
        }
//        Assert.notNull(splitPageInfo, "splitPageInfo 不能为空!");
        BaseMapper<T> pomapper = this.convertToMapper(po);
        SqlAndParameters<Map<String, Object>> sp = pomapper.getSelectSql_(where, parameters);
        return this.selectSplit(sp.getSql(), sp.getParameters(), currentPage, pageSize, po);
    }

    @Override
    public <T extends BasePo<T>> GenericPager<T> selectSplit(T po, int currentPage, int pageSize, Sorts.Sort sort) throws DataAccessException{
        Assert.notNull(po, "po 不能为空!");
        Object[] result = JdbcUtils.acquireQueryCondition(po);
        BaseMapper pomapper = this.convertToMapper(po);
        SqlAndParameters<Map<String, Object>> sp = pomapper.getSelectSql_(result[0].toString(), (Map<String, Object>)result[1]);
        if(logger.isDebugEnabled()){
            logger.debug(sp.getSql());
            logger.debug("fieldNameAndValue={}", result[1]);
        }
        String sql = sort == null? sp.getSql() : this.paginationHelper.getOrderBySql(sp.getSql(), sort);
        return this.selectSplit(sql, sp.getParameters(), currentPage, pageSize, po);
//        return this.selectSplit(sp.getSql(), sp.getParameters(), currentPage, pageSize, po);
    }

    @Override
    public <T extends BasePo<T>> GenericPager<T> selectSplit(T po, int currentPage, int pageSize) throws DataAccessException{
        return this.selectSplit(po, currentPage, pageSize, (Sorts.Sort) null);
    }

    /**
     * @see com.walker.jdbc.BaseDao#select(String, Object[])
     */
    @Override
    public List<Map<String, Object>> select(String sql, Object[] parameters) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");
        printSql(new SqlAndParameters<>(sql, parameters));
        return this.jdbcTemplate.query(sql, new ColumnMapRowMapper(), parameters);
    }

    /**
     * @see com.walker.jdbc.BaseDao#select(String, Object[], RowMapper<T> )
     */
    @Override
    public <T> List<T> select(String sql, Object[] parameters, RowMapper<T> rowMapper) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");

        printSql(new SqlAndParameters<>(sql, parameters));
        return this.jdbcTemplate.query(sql, rowMapper, parameters);
    }
    /**
     * @see com.walker.jdbc.BaseDao#select(String, Object[], BasePo)
     */
    @Override
    public <T extends BasePo<T>> List<T> select(String sql, Object[] parameters, T po) throws DataAccessException {
        return this.select(sql, parameters, this.convertToMapper(po));
    }

    /**
     * @see com.walker.jdbc.BaseDao#selectSplit(String, Object[], int, int)
     */
    @Override
    public MapPager selectSplit(String sql, Object[] parameters, int currentPage, int pageSize, Sorts.Sort sort) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空!");
//        Assert.notNull(parameters, "parameters 不能为空!");
//        Assert.notNull(splitPageInfo, "splitPageInfo 不能为空!");
        if(parameters == null){
            parameters = new Object[]{};
        }
        int cnt = this.queryForInt("select count(1) from (" + sql + ")  total", parameters);
        MapPager pager = ListPageContext.createMapPager(null, currentPage, pageSize, cnt);
//        splitPageInfo.setTotalRowsAmount(cnt);
//        SqlAndParameters<Object[]> sp = this.paginationHelper.getPaginationSql(sql, parameters, splitPageInfo);
        SqlAndParameters<Object[]> sp = this.paginationHelper.getSqlPagingQuery(sql, parameters, pager, sort);
        printSql(sp);
        List<Map<String, Object>> data = this.jdbcTemplate.query(sp.getSql(), new ColumnMapRowMapper(), sp.getParameters());
        pager.setDatas(data);
        return pager;
    }

    @Override
    public MapPager selectSplit(String sql, Object[] parameters, int currentPage, int pageSize) throws DataAccessException {
        return this.selectSplit(sql,parameters, currentPage, pageSize, (Sorts.Sort) null);
    }

    /**
     * @see com.walker.jdbc.BaseDao#selectSplit(String, Object[], int, int, RowMapper<T> )
     */
    @Override
    public <T> GenericPager<T> selectSplit(String sql, Object[] parameters, int currentPage, int pageSize
            , RowMapper<T> rowMapper, Sorts.Sort sort) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空!");
//        Assert.notNull(parameters, "parameters 不能为空!");
        Assert.notNull(rowMapper, "rowMapper 不能为空!");
        if(parameters == null){
            parameters = new Object[]{};
        }
        int cnt = this.queryForInt("select count(1) from (" + sql + ") total", parameters);
        GenericPager<T> pager = ListPageContext.createGenericPager(currentPage, pageSize, cnt);
//        splitPageInfo.setTotalRowsAmount(cnt);

//        SqlAndParameters<Object[]> sp = this.paginationHelper.getPaginationSql(sql, parameters, splitPageInfo);
        SqlAndParameters<Object[]> sp = this.paginationHelper.getSqlPagingQuery(sql, parameters, pager, sort);
        printSql(sp);
        List<T> data = this.jdbcTemplate.query(sp.getSql(), rowMapper, sp.getParameters());
        pager.setDatas(data);
        return pager;
    }
    @Override
    public <T> GenericPager<T> selectSplit(String sql, Object[] parameters, int currentPage, int pageSize, RowMapper<T> rowMapper) throws DataAccessException{
        return this.selectSplit(sql, parameters, currentPage, pageSize, rowMapper, null);
    }

    @Override
    public <T extends BasePo<T>> GenericPager<T> selectSplit(String sql
            , Object[] parameters, int currentPage, int pageSize, T po) throws DataAccessException {
        return this.selectSplit(sql, parameters, currentPage, pageSize, this.convertToMapper(po));
    }

    /**
     * @see com.walker.jdbc.BaseDao#select(String, Map< String, Object> )
     */
    @Override
    public List<Map<String, Object>> select(String sql, Map<String, Object> parameters) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");
        printSql(new SqlAndParameters<>(sql, parameters));
        return this.namedParameterJdbcTemplate.query(sql, parameters, new ColumnMapRowMapper());
    }

    /**
     * @see com.walker.jdbc.BaseDao#select(String, Map< String, Object>, RowMapper<T> )
     */
    @Override
    public <T> List<T> select(String sql, Map<String, Object> parameters, RowMapper<T> mapper) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");
        printSql(new SqlAndParameters<>(sql, parameters));
        return this.namedParameterJdbcTemplate.query(sql, parameters, mapper);
    }

    /**
     * @see com.walker.jdbc.BaseDao#select(String, Map< String, Object>, BasePo<T> )
     */
    @Override
    public <T extends BasePo<T>> List<T> select(String sql, Map<String, Object> parameters, T po) throws DataAccessException {
        return this.select(sql, parameters, this.convertToMapper(po));
    }

    /**
     * @see com.walker.jdbc.BaseDao#selectSplit(String, Map, int, int)
     */
    @Override
    public MapPager selectSplit(String sql, Map<String, Object> parameters, int currentPage, int pageSize, Sorts.Sort sort) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空!");
//        Assert.notNull(parameters, "parameters 不能为空!");
        if(parameters == null){
            parameters = new HashMap<>(2);
        }

        int cnt = this.queryForInt("select count(1) from (" + sql + ") total", parameters);
//        splitPageInfo.setTotalRowsAmount(cnt);
        MapPager pager = ListPageContext.createMapPager(null, currentPage, pageSize, cnt);
//        SqlAndParameters<Map<String, Object>> sp = this.paginationHelper.getPaginationSql(sql, parameters, splitPageInfo);
        SqlAndParameters<Map<String, Object>> sp = this.paginationHelper.getSqlPagingQuery(sql, parameters, pager, sort);
        printSql(sp);
        List<Map<String, Object>> data = this.namedParameterJdbcTemplate.query(sp.getSql(), sp.getParameters(), new ColumnMapRowMapper());
        pager.setDatas(data);
        return pager;
    }

    @Override
    public MapPager selectSplit(String sql, Map<String, Object> parameters, int currentPage, int pageSize) throws DataAccessException {
        return this.selectSplit(sql, parameters, currentPage, pageSize, (Sorts.Sort) null);
    }

    /**
     * @see com.walker.jdbc.BaseDao#selectSplit(String, Map, int, int, RowMapper)
     */
    @Override
    public <T> GenericPager<T> selectSplit(String sql, Map<String, Object> parameters
            , int currentPage, int pageSize, RowMapper<T> rowMapper, Sorts.Sort sort) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空!");
//        Assert.notNull(parameters, "parameters 不能为空!");
        Assert.notNull(rowMapper, "rowMapper 不能为空!");
        if(parameters == null){
            parameters = new HashMap<>(2);
        }
        int cnt = this.queryForInt("select count(1) from (" + sql + ") total", parameters);
//        splitPageInfo.setTotalRowsAmount(cnt);
        GenericPager<T> pager = ListPageContext.createGenericPager(currentPage, pageSize, cnt);

//        SqlAndParameters<Map<String, Object>> sp = this.paginationHelper.getPaginationSql(sql, parameters, splitPageInfo);
        SqlAndParameters<Map<String, Object>> sp = this.paginationHelper.getSqlPagingQuery(sql, parameters, pager, sort);
        printSql(sp);
        List<T> data = this.namedParameterJdbcTemplate.query(sp.getSql(), sp.getParameters(), rowMapper);
        pager.setDatas(data);
        return pager;
    }
    @Override
    public <T> GenericPager<T> selectSplit(String sql, Map<String, Object> parameters
            , RowMapper<T> rowMapper, Sorts.Sort sort) throws DataAccessException {
        // 2023-10-24 前后端界面使用:PageSearch分页参数，如果存在优先使用。
        PageSearch pageSearch = ListPageContext.getPageSearch();
        if(pageSearch != null){
            return this.selectSplit(sql, parameters, pageSearch.getPageIndex(), pageSearch.getPageSize(), rowMapper, sort);
        } else {
            return this.selectSplit(sql, parameters, ListPageContext.getCurrentPageIndex(), ListPageContext.getCurrentPageSize(), rowMapper, sort);
        }
    }

    @Override
    public <T> GenericPager<T> selectSplit(String sql
            , Map<String, Object> parameters, int currentPage, int pageSize, RowMapper<T> rowMapper) throws DataAccessException {
        return selectSplit(sql, parameters, currentPage, pageSize, rowMapper, null);
    }

    /**
     * @see BaseDao#selectSplit(String, Map, int, int, BasePo)
     */
    @Override
    public <T extends BasePo<T>> GenericPager<T> selectSplit(String sql
            , Map<String, Object> parameters, int currentPage, int pageSize, T po) throws DataAccessException {
        return this.selectSplit(sql, parameters, currentPage, pageSize, this.convertToMapper(po));
    }

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    //~ 以下方法简化分页参数
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    @Override
    public <T extends BasePo<T>> GenericPager<T> selectSplit(String sql
            , Map<String, Object> parameters, T po) throws DataAccessException{
        // 2023-04-01 前后端界面使用:PageSearch分页参数，如果存在优先使用。
        PageSearch pageSearch = ListPageContext.getPageSearch();
        if(pageSearch != null){
            return this.selectSplit(sql, parameters, pageSearch.getPageIndex(), pageSearch.getPageSize(), po);
        } else {
            return this.selectSplit(sql, parameters, ListPageContext.getCurrentPageIndex(), ListPageContext.getCurrentPageSize(), po);
        }
//        return this.selectSplit(sql, parameters, ListPageContext.getCurrentPageIndex(), ListPageContext.getCurrentPageSize(), po);
    }

    /**
     * 分页查询，根据PO里面字段，做简单条件查询。
     * @param po
     * @return
     * @param <T>
     * @throws DataAccessException
     * @date 2023-04-21
     */
    @Override
    public <T extends BasePo<T>> GenericPager<T> selectSplit(T po, Sorts.Sort sort) throws DataAccessException{
        PageSearch pageSearch = ListPageContext.getPageSearch();
        if(pageSearch != null){
            return this.selectSplit(po, pageSearch.getPageIndex(), pageSearch.getPageSize(), sort);
        } else {
            return this.selectSplit(po, ListPageContext.getCurrentPageIndex(), ListPageContext.getCurrentPageSize(), sort);
        }
    }

    @Override
    public <T extends BasePo<T>> GenericPager<T> selectSplit(T po) throws DataAccessException{
        return this.selectSplit(po, null);
    }

    @Override
    public <T extends BasePo<T>> GenericPager<T> selectSplit(String sql
            , Object[] parameters, T po) throws DataAccessException{
        // 2023-04-01 前后端界面使用:PageSearch分页参数，如果存在优先使用。
        PageSearch pageSearch = ListPageContext.getPageSearch();
        if(pageSearch != null){
            return this.selectSplit(sql, parameters, pageSearch.getPageIndex(), pageSearch.getPageSize(), po);
        } else {
            return this.selectSplit(sql, parameters, ListPageContext.getCurrentPageIndex(), ListPageContext.getCurrentPageSize(), po);
        }
//        return this.selectSplit(sql, parameters, ListPageContext.getCurrentPageIndex(), ListPageContext.getCurrentPageSize(), po);
    }

    @Override
    public <T> GenericPager<T> selectSplit(String sql, Object[] parameters, RowMapper<T> rowMapper) throws DataAccessException{
        // 2023-04-01 前后端界面使用:PageSearch分页参数，如果存在优先使用。
        PageSearch pageSearch = ListPageContext.getPageSearch();
        if(pageSearch != null){
            return this.selectSplit(sql, parameters, pageSearch.getPageIndex(), pageSearch.getPageSize(),rowMapper);
        } else {
            return this.selectSplit(sql, parameters, ListPageContext.getCurrentPageIndex(), ListPageContext.getCurrentPageSize(), rowMapper);
        }
//        return this.selectSplit(sql, parameters, ListPageContext.getCurrentPageIndex(), ListPageContext.getCurrentPageSize(), rowMapper);
    }

    @Override
    public MapPager selectSplit(String sql, Object[] parameters) throws DataAccessException{
        // 2023-04-01 前后端界面使用:PageSearch分页参数，如果存在优先使用。
        PageSearch pageSearch = ListPageContext.getPageSearch();
        if(pageSearch != null){
            return this.selectSplit(sql, parameters, pageSearch.getPageIndex(), pageSearch.getPageSize());
        } else {
            return this.selectSplit(sql, parameters, ListPageContext.getCurrentPageIndex(), ListPageContext.getCurrentPageSize());
        }
//        return this.selectSplit(sql, parameters, ListPageContext.getCurrentPageIndex(), ListPageContext.getCurrentPageSize());
    }

    @Override
    public <T extends BasePo<T>> GenericPager<T> selectSplit(T po
            , String where, Map<String, Object> parameters) throws DataAccessException{
        // 2023-04-01 前后端界面使用:PageSearch分页参数，如果存在优先使用。
        PageSearch pageSearch = ListPageContext.getPageSearch();
        if(pageSearch != null){
            return this.selectSplit(po, where, parameters, pageSearch.getPageIndex(), pageSearch.getPageSize());
        } else {
            return this.selectSplit(po, where, parameters, ListPageContext.getCurrentPageIndex(), ListPageContext.getCurrentPageSize());
        }
    }

//    /**
//     * @see com.walker.jdbc.BaseDao#selectSplitEx(String, Map, SplitPageInfo, T)
//     */
//    @Override
//    public <T extends BasePo<T>> SplitPageInfo selectSplitEx(String sql, Map<String, Object> parameters, SplitPageInfo splitPageInfo, T po) throws DataAccessException {
//        return this.selectSplitEx(sql, parameters, splitPageInfo, this.convertToMapper(po));
//    }

    /**
     * @see com.walker.jdbc.BaseDao#execute(String, Object[])
     */
    @Override
    public int execute(String sql, Object[] parameters) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");
        printSql(new SqlAndParameters<>(sql, parameters));
        return this.jdbcTemplate.update(sql, parameters);
    }

    /**
     * @see com.walker.jdbc.BaseDao#execute(String, Map< String, Object> )
     */
    @Override
    public int execute(String sql, Map<String, Object> parameters) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");

        printSql(new SqlAndParameters<>(sql, parameters));
        return this.namedParameterJdbcTemplate.update(sql, parameters);
    }

    /**
     * @see com.walker.jdbc.BaseDao#queryForInt(String, Object[])
     */
    @Override
    public int queryForInt(String sql, Object[] parameters) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");

        Integer res = this.queryForObject(sql, parameters, Integer.class);
        if (res == null) {
            throw new PersistenceException("查询结果为空，不能转换为int");
        }
        return res;
    }

    /**
     * @see com.walker.jdbc.BaseDao#queryForInt(String, Map< String, Object> )
     */
    @Override
    public int queryForInt(String sql, Map<String, Object> parameters) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");

        Integer res = this.queryForObject(sql, parameters, Integer.class);
        if (res == null) {
            throw new PersistenceException("查询结果为空，不能转换为int");
        }
        return res;
    }

    /**
     * @see com.walker.jdbc.BaseDao#queryForObject(String, Object[], Class<T> )
     */
    @Override
    public <T> T queryForObject(String sql, Object[] parameters, Class<T> clazz) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");

        List<T> list = this.select(sql, parameters, new SingleColumnRowMapper<T>(clazz));
        return getUnique(list);
    }

    private <T> T getUnique(List<T> list) {
        if (StringUtils.isEmptyList(list)) {
            return null;
        } else if (list.size() > 1) {
            throw new PersistenceException("期望返回一条记录，实际返回 " + list.size() + " 条记录！");
        } else {
            return list.get(0);
        }
    }

    /**
     * @see com.walker.jdbc.BaseDao#queryForObject(String, Map, Class)
     */
    @Override
    public <T> T queryForObject(String sql, Map<String, Object> parameters, Class<T> clazz) throws DataAccessException {
        Assert.notNull(sql, "sql 不能为空!");
        Assert.notNull(parameters, "parameters 不能为空!");

        List<T> list = this.select(sql, parameters, new SingleColumnRowMapper<T>(clazz));
        return getUnique(list);
    }

    /**
     * @see com.walker.jdbc.BaseDao#execCall(String, Object[])
     */
    @Override
    public void execCall(final String functionName, final Object[] parameters) throws DataAccessException {
        Assert.notNull(functionName, "functionName 不能为空!");
        printSql(new SqlAndParameters<>(functionName, parameters));
        this.jdbcTemplate.execute(new CallableStatementCreator() {
            @Override
            public CallableStatement createCallableStatement(Connection con) throws SQLException {
                CallableStatement cs = con.prepareCall(functionName);
                if (parameters != null) {
                    for (int i = 0; i < parameters.length; i++) {
                        cs.setObject(i + 1, parameters[i]);
                    }
                }
                return cs;
            }
        }, new CallableStatementCallback() {
            @Override
            public Object doInCallableStatement(CallableStatement cs) throws SQLException, DataAccessException {
                cs.execute();
                return null;
            }
        });
    }

    /**
     * @see com.walker.jdbc.BaseDao#execCall(String, Object[], Class<T> )
     */
    @Override
    public <T> T execCall(final String functionName, final Object[] parameters, final Class<T> clazz) throws DataAccessException {
        Assert.notNull(functionName, "functionName 不能为空!");

        printReturnCall(functionName, parameters);
        return this.jdbcTemplate.execute(new CallableStatementCreator() {
            @Override
            public CallableStatement createCallableStatement(Connection con) throws SQLException {
                CallableStatement cs = con.prepareCall(functionName);
                cs.registerOutParameter(1, Types.NVARCHAR);
                if (parameters != null) {
                    for (int i = 0; i < parameters.length; i++) {
                        cs.setObject(i + 2, parameters[i]);
                    }
                }
                return cs;
            }
        }, new CallableStatementCallback<T>() {
            @Override
            public T doInCallableStatement(CallableStatement cs) throws SQLException, DataAccessException {
                cs.execute();
                cs.getResultSet();
                return (T) getStatementValue(cs, 1, clazz);
            }
        });
    }

    private void printReturnCall(String functionName, Object[] parameters) {
        if (this.isShowSql()) {
            List<Object> ps = new ArrayList<>();
            ps.add("?");
            if (parameters != null) {
                Collections.addAll(ps, parameters);
            }
            Matcher matcher = ARRAY_PARAMETER_PLACEHOLDER.matcher(functionName);
            StringBuffer sb = new StringBuffer();
            int i = 0;
            while (matcher.find()) {
                matcher.appendReplacement(sb, getSqlParamValue(ps.get(i++)));
            }
            matcher.appendTail(sb);
            logger.info("{}", sb);
        }
    }

    private String getSqlParamValue(Object value) {
        if (value == null) {
            return "null";
        }
        if (value instanceof String || value instanceof java.util.Date) {
            return "'" + StringSqlUtils.escapeSql(value.toString()) + "'";
        } else {
            return value.toString();
        }
    }

    /**
     * 获取返回值
     *
     * @param cs           存储过程调用语句
     * @param index        索引
     * @param requiredType 类型
     * @return requiredType类型数据
     * @throws SQLException sql异常
     */
    private Object getStatementValue(CallableStatement cs, int index, Class<?> requiredType) throws SQLException {
        Object value;
        // Explicitly extract typed value, as far as possible.
        if (String.class == requiredType) {
            return cs.getString(index);
        } else if (boolean.class == requiredType || Boolean.class == requiredType) {
            value = cs.getBoolean(index);
        } else if (byte.class == requiredType || Byte.class == requiredType) {
            value = cs.getByte(index);
        } else if (short.class == requiredType || Short.class == requiredType) {
            value = cs.getShort(index);
        } else if (int.class == requiredType || Integer.class == requiredType) {
            value = cs.getInt(index);
        } else if (long.class == requiredType || Long.class == requiredType) {
            value = cs.getLong(index);
        } else if (float.class == requiredType || Float.class == requiredType) {
            value = cs.getFloat(index);
        } else if (double.class == requiredType || Double.class == requiredType || Number.class == requiredType) {
            value = cs.getDouble(index);
        } else if (BigDecimal.class == requiredType) {
            return cs.getBigDecimal(index);
        } else if (java.sql.Date.class == requiredType) {
            return cs.getDate(index);
        } else if (Time.class == requiredType) {
            return cs.getTime(index);
        } else if (Timestamp.class == requiredType || java.util.Date.class == requiredType) {
            return cs.getTimestamp(index);
        } else if (byte[].class == requiredType) {
            return cs.getBytes(index);
        } else if (Blob.class == requiredType) {
            return cs.getBlob(index);
        } else if (Clob.class == requiredType) {
            return cs.getClob(index);
        } else {
            throw new PersistenceException("不支持的类型" + requiredType + "！");
        }
        return value;
    }

    /**
     * @see com.walker.jdbc.BaseDao#execBatchUpdate(String, List<?> )
     */
    @Override
    public int execBatchUpdate(String sql, List<?> parametersList) throws DataAccessException {
        Assert.notNull(sql, "sql不能为空！");
        Assert.notNull(parametersList, "parametersList不能为空！");
        if (parametersList.size() == 0) {
            throw new PersistenceException("parametersList不能为空！");
        }

        int[] temp;
        if (parametersList.get(0) instanceof Map) {
            Map<String, Object>[] params = parametersList.toArray(new Map[parametersList.size()]);
            temp = this.namedParameterJdbcTemplate.batchUpdate(sql, params);
        } else if (parametersList.get(0).getClass().isArray()) {
            temp = this.jdbcTemplate.batchUpdate(sql, (List<Object[]>) parametersList);
        } else {
            throw new PersistenceException("parametersList内容类型参数不合法！");
        }

        int result = 0;
        if (temp != null) {
            for (int v : temp) {
                result += v;
            }
        }
        return result;
    }

    @Override
    public void update(String sql) {

    }

    @Override
    public int update(String sql, Object[] args) {
        return 0;
    }

    @Override
    public <T> List<T> sqlQuery(String sql, RowMapper<T> rowMapper) {
        return null;
    }

    @Override
    public <T> List<T> sqlQuery(String sql, Object[] args, RowMapper<T> rowMapper) {
        return null;
    }

    @Override
    public List<Map<String, Object>> sqlQueryListMap(String sql, Object[] args) {
        return null;
    }

    @Override
    public <E> GenericPager<E> sqlGeneralQueryPager(String sql, Object[] args, RowMapper<E> rowMapper) {
        return null;
    }

    @Override
    public <E> GenericPager<E> sqlGeneralQueryPager(String sql, Object[] args, RowMapper<E> rowMapper, int pageIndex) {
        return null;
    }

    @Override
    public <T> GenericPager<T> sqlGeneralQueryPager(String sql, Object[] args, RowMapper<T> rowMapper, int pageIndex, int pageSize) {
        return null;
    }

    @Override
    public <T> T sqlMathQuery(String sql, Object[] args, Class<T> clazz) {
        return null;
    }

    @Override
    public <T> List<T> sqlListObjectWhereIn(String sql, RowMapper<T> rowMapper, SqlParameterSource paramSource) {
        return null;
    }

    @Override
    public List<Map<String, Object>> queryListObjectWhereIn(String sql, SqlParameterSource paramSource) {
        return null;
    }

    @Override
    public <T> void execute(final String sql, final Object[] parameters, final RowMapper<T> rowMapper, final RowExecution<T> rowExecution) {
        this.jdbcTemplate.execute(sql, new PreparedStatementCallback<T>() {
            @Override
            public T doInPreparedStatement(PreparedStatement ps) throws SQLException, DataAccessException {
                ps.setFetchSize(512);
                if (parameters != null) {
                    for (int i = 0; i < parameters.length; i++) {
                        ps.setObject(i + 1, parameters[i]);
                    }
                }
                printSql(new SqlAndParameters<>(sql, parameters));
                ResultSet rs = ps.executeQuery();
                try {
                    while (rs.next()) {
                        T t = rowMapper.mapRow(rs, 1);
                        rowExecution.execute(t);
                    }
                } catch (Exception e) {
                    throw new PersistenceException(e);
                } finally {
                    org.springframework.jdbc.support.JdbcUtils.closeResultSet(rs);
                }
                return null;
            }
        });
    }

    @Override
    public <T extends BasePo<T>> void execute(final String sql, final Object[] parameters, final T po, final RowExecution<T> rowExecution) {
        this.execute(sql, parameters, this.convertToMapper(po), rowExecution);
    }
}
