package com.walker.jdbc.ds;

import com.walker.db.DatabaseType;
import com.walker.infrastructure.ApplicationRuntimeException;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.jdbc.Constants;
import com.walker.jdbc.DataSourceMeta;
import com.walker.jdbc.util.TextUtils;
import com.zaxxer.hikari.HikariDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 系统默认自定义数据源对象，便于获得一些必要元数据。
 * @date 2022-08-12
 * @author 时克英
 */
public class DefaultDataSource extends HikariDataSource implements DataSourceMeta {

    protected final transient Logger logger = LoggerFactory.getLogger(this.getClass());

//    private String url;
    private DatabaseType databaseType = null;

    private String username;

    private String password;

    private String databaseName;

    @Override
    public String getUsername() {
        return username;
    }

    public void setUsername(String username) {
        this.username = username;
    }

    @Override
    public String getPassword() {
        return password;
    }

    @Override
    public void setPassword(String password) {
        this.password = password;
    }

    @Override
    public DatabaseType getDatabaseType() {
        return this.databaseType;
    }

    @Override
    public String getUrl() {
        return this.getJdbcUrl();
    }

    @Override
    public String getDatabaseName() {
        return this.databaseName;
    }

    @Override
    public String getIp(){
        return this.ip;
    }

    @Override
    public int getPort(){
        return this.port;
    }

    @Override
    public void setJdbcUrl(String jdbcUrl) {
        super.setJdbcUrl(jdbcUrl);
//        this.url = jdbcUrl;
        logger.info("jdbcUrl = " + jdbcUrl);

        // 解析ip、端口、数据库名称
        String[] ipAndPortAndService = TextUtils.parseIpAndPortFromUrl(jdbcUrl);
        if(ipAndPortAndService != null){
            this.ip = ipAndPortAndService[0];
            this.port = Integer.parseInt(ipAndPortAndService[1]);
            this.databaseName = ipAndPortAndService[2];
        }

        String[] values = jdbcUrl.split(StringUtils.SEPARATOR_COLON);
        if(values == null || values.length <=1){
            throw new ApplicationRuntimeException("DataSource set jdbcUrl error: " + jdbcUrl);
        }

        for(String v : values){
            if(v.equalsIgnoreCase(DatabaseType.NAME_MYSQL)){
                this.databaseType = DatabaseType.MYSQL;
                return;
            } else if(v.equalsIgnoreCase(DatabaseType.NAME_ORACLE)){
                this.databaseType = DatabaseType.ORACLE;
                return;
            } else if(v.equalsIgnoreCase(DatabaseType.NAME_POSTGRES)){
                this.databaseType = DatabaseType.POSTGRES;
                return;
            } else if(v.equalsIgnoreCase(DatabaseType.NAME_SQLITE)){
                this.databaseType = DatabaseType.SQLITE;
                return;
            } else if(v.equalsIgnoreCase(DatabaseType.NAME_DAMENG)){
                this.databaseType = DatabaseType.DAMENG;
                return;
            } else {
                //
            }
        }

        if(this.databaseType == null){
            throw new UnsupportedOperationException("不支持的数据库URL参数：" + jdbcUrl);
        }
        logger.info("数据库类型：" + this.databaseType.name());
    }

    @Override
    public void setDriverClassName(String driverClassName){
        super.setDriverClassName(driverClassName);
        logger.debug("driverClassName = " + driverClassName);
    }

    //=============================================================================================
    // 以下代码用于手动创建独立数据源使用，比如：采集模块动态创建: DatabaseConnector，2022-09-09，时克英
    //=============================================================================================

    protected String ip;
    protected int port = 0;

    public void setDatabaseType(int databaseTypeValue) {
        this.databaseType = DatabaseType.getType(databaseTypeValue);
        String driverName = Constants.getDriverClassName(this.databaseType);
//        String driverName = null;
//        if(databaseType == DatabaseType.DERBY){
//            driverName = Constants.DRIVER_NAME_DERBY;
//        } else if(databaseType == DatabaseType.MYSQL){
//            driverName = Constants.DRIVER_NAME_MYSQL;
//        } else if(databaseType == DatabaseType.ORACLE){
//            driverName = Constants.DRIVER_NAME_ORACLE;
//        } else if(databaseType == DatabaseType.POSTGRES){
//            driverName = Constants.DRIVER_NAME_POSTGRES;
//        } else if(databaseType == DatabaseType.SQLSERVER){
//            driverName = Constants.DRIVER_NAME_SQLSERVER;
//        } else if(databaseType == DatabaseType.DAMENG){
//            driverName = Constants.DRIVER_NAME_DAMENG;
//        } else
//            throw new IllegalArgumentException("unsupported database type: " + databaseTypeValue);
        try {
            this.setDriverClassName(driverName);
        } catch (Exception e) {
            throw new IllegalArgumentException("未找到该驱动：" + driverName, e);
        }
    }

    public void setIp(String ip) {
        assert (StringUtils.isNotEmpty(ip));
        this.ip = ip;
    }

    public void setPort(int port) {
        assert(port > 0 && port < Integer.MAX_VALUE);
        this.port = port;
    }

    public void setDatabaseName(String databaseName) {
        if(databaseType == null){
            throw new IllegalArgumentException("databaseType must be set firstly!");
        }
        if(databaseType != DatabaseType.DERBY && StringUtils.isEmpty(ip)){
            throw new IllegalArgumentException("ip or port must be set firstly!");
        }
        assert (StringUtils.isNotEmpty(databaseName));
//		this.databaseName = databaseName;

//        StringBuilder url = new StringBuilder();
//        if(databaseType == DatabaseType.DERBY){
//            url.append("jdbc:derby:").append(databaseName);
//
//        } else if(databaseType == DatabaseType.MYSQL){
//            url.append("jdbc:mysql://").append(ip).append(":")
//                    .append(port == 0 ? Constants.DEFAULT_PORT_MYSQL : port)
//                    .append("/").append(databaseName);
//            // 因为oracle不支持这样设置参数，所以位置调换到这里 2016-11-30
//            if(url.indexOf("?") >= 0){
////                url.append("&amp;autoReconnect=true");		//当数据库连接异常中断时，是否自动重新连接
////                url.append("&amp;failOverReadOnly=false");	//自动重连成功后，连接是否设置为只读
//                url.append("&autoReconnect=true");		//当数据库连接异常中断时，是否自动重新连接
//                url.append("&failOverReadOnly=false");	//自动重连成功后，连接是否设置为只读
//            } else {
//                url.append("?autoReconnect=true");
////                url.append("&amp;failOverReadOnly=false");
//                url.append("&failOverReadOnly=false");
//            }
//
//        } else if(databaseType == DatabaseType.ORACLE){
//            url.append("jdbc:oracle:thin:@").append(ip).append(":")
//                    .append(port == 0 ? Constants.DEFAULT_PORT_ORACLE : port)
//                    .append(":").append(databaseName);
//
//        } else if(databaseType == DatabaseType.POSTGRES){
//            url.append("jdbc:postgresql://").append(ip).append(":")
//                    .append(port == 0 ? Constants.DEFAULT_PORT_POSTGRES : port)
//                    .append("/").append(databaseName);
//
//        } else if(databaseType == DatabaseType.SQLSERVER){
//            url.append("jdbc:sqlserver://").append(ip).append(":")
//                    .append(port == 0 ? Constants.DEFAULT_PORT_POSTGRES : port)
//                    .append(";DatabaseName=").append(databaseName);
//
//        } else if(databaseType == DatabaseType.DAMENG){
//            url.append("jdbc:dm://").append(ip).append(":")
//                    .append(port == 0 ? Constants.DEFAULT_PORT_POSTGRES : port)
//                    .append("?SCHEMA=").append(databaseName);
//        } else
//            throw new IllegalArgumentException("unsupported database type: " + databaseType);
        String url = Constants.getJdbcUrl(this.databaseType, databaseName, this.ip, this.port);
        this.setJdbcUrl(url);

        //=========================================
        // 创建本地连接数据引擎，20171213，时克英
        //=========================================
//		if(localDatabaseStartMode){
//			address = new Address();
//			address.setUrl(ip);
//			address.setPort(port);
//			address.setService(databaseName);
//			if(StringUtils.isEmpty(this.getUser()) || StringUtils.isEmpty(this.getPassword())){
//				throw new IllegalArgumentException("数据库未配置用户名、密码");
//			}
//			address.setAuthentication(new String(this.getUser().getBytes()));
//			address.setCertification(new String(this.getPassword().getBytes()));
//
//			LocalDatabaseMetaEngine localDatabaseMetaEngine = new LocalDatabaseMetaEngine();
//			localDatabaseMetaEngine.setAddress(databaseType, address);
//
//			MyApplicationConfig.setLocalDatabaseMetaEngine(localDatabaseMetaEngine);
//		}

    }
}
