package com.walker.jdbc.ds;

import com.alibaba.druid.pool.DruidDataSource;
import com.walker.db.DatabaseType;
import com.walker.infrastructure.ApplicationRuntimeException;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.jdbc.Constants;
import com.walker.jdbc.DataSourceMeta;
import com.walker.jdbc.util.TextUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 包装阿里 <code>DruidDataSource</code> 数据源，形成自己的类型，隔离依赖。
 * @author 时克英
 * @date 2023-03-15
 */
public class MyDruidDataSource extends DruidDataSource implements DataSourceMeta {

    protected final transient Logger logger = LoggerFactory.getLogger(this.getClass());

    @Override
    public DatabaseType getDatabaseType() {
        return this.databaseType;
    }

    @Override
    public String getDatabaseName() {
        return this.databaseName;
    }

    @Override
    public String getIp() {
        return this.ip;
    }

    @Override
    public int getPort() {
        return this.port;
    }

    @Override
    public String getUsername() {
        return username;
    }
    @Override
    public String getPassword() {
        return password;
    }
    @Override
    public String getUrl() {
        return this.jdbcUrl;
    }

    @Override
    public void setPassword(String password) {
        this.password = password;
    }
    @Override
    public void setUsername(String username) {
        this.username = username;
    }

    /**
     * 重写druid设置url方法，初始化数据源。
     * @param jdbcUrl
     * @date 2023-03-15
     */
    @Override
    public void setUrl(String jdbcUrl) {
        super.setUrl(jdbcUrl);
        logger.info("jdbcUrl = " + jdbcUrl);

        // 解析ip、端口、数据库名称
        String[] ipAndPortAndService = TextUtils.parseIpAndPortFromUrl(jdbcUrl);
        if(ipAndPortAndService != null){
            this.ip = ipAndPortAndService[0];
            this.port = Integer.parseInt(ipAndPortAndService[1]);
            this.databaseName = ipAndPortAndService[2];
        }

        String[] values = jdbcUrl.split(StringUtils.SEPARATOR_COLON);
        if(values == null || values.length <=1){
            throw new ApplicationRuntimeException("DataSource set jdbcUrl error: " + jdbcUrl);
        }

        for(String v : values){
            if(v.equalsIgnoreCase(DatabaseType.NAME_MYSQL)){
                this.databaseType = DatabaseType.MYSQL;
                return;
            } else if(v.equalsIgnoreCase(DatabaseType.NAME_ORACLE)){
                this.databaseType = DatabaseType.ORACLE;
                return;
            } else if(v.equalsIgnoreCase(DatabaseType.NAME_POSTGRES) || v.equalsIgnoreCase(DatabaseType.NAME_POSTGRESQL)){
                this.databaseType = DatabaseType.POSTGRES;
                return;
            } else if(v.equalsIgnoreCase(DatabaseType.NAME_SQLITE)){
                this.databaseType = DatabaseType.SQLITE;
                return;
            } else if(v.equalsIgnoreCase(DatabaseType.NAME_DAMENG)){
                this.databaseType = DatabaseType.DAMENG;
                return;
            } else {
                //
            }
        }

        if(this.databaseType == null){
            throw new UnsupportedOperationException("不支持的数据库URL参数：" + jdbcUrl);
        }
        logger.info("数据库类型：" + this.databaseType.name());
    }

    /**
     * 重写druid设置url方法，初始化数据源。
     * @param driverClassName
     * @date 2023-03-15
     */
    @Override
    public void setDriverClassName(String driverClassName){
        super.setDriverClassName(driverClassName);
        logger.debug("driverClassName = " + driverClassName);
    }

    private DatabaseType databaseType = null;
    private String username;
    private String password;
    private String databaseName;

    //=============================================================================================
    // 以下代码用于手动创建独立数据源使用，比如：采集模块动态创建: DatabaseConnector，2022-09-09，时克英
    //=============================================================================================

    protected String ip;
    protected int port = 0;

    public void setIp(String ip) {
        assert (StringUtils.isNotEmpty(ip));
        this.ip = ip;
    }

    public void setPort(int port) {
        assert(port > 0 && port < Integer.MAX_VALUE);
        this.port = port;
    }

    public void setDatabaseType(int databaseTypeValue) {
        this.databaseType = DatabaseType.getType(databaseTypeValue);
        String driverName = Constants.getDriverClassName(this.databaseType);
        this.setDriverClassName(driverName);
    }

    public void setDatabaseName(String databaseName) {
        if(StringUtils.isEmpty(databaseName)){
            throw new IllegalArgumentException("databaseName 必须填写!");
        }
        if(databaseType == null){
            throw new IllegalArgumentException("databaseType must be set firstly!");
        }
        if(databaseType != DatabaseType.DERBY && StringUtils.isEmpty(ip)){
            throw new IllegalArgumentException("ip or port must be set firstly!");
        }
        String url = Constants.getJdbcUrl(this.databaseType, databaseName, this.ip, this.port);
        this.setUrl(url);
    }
}
