package com.walker.jdbc.util;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.walker.infrastructure.ApplicationRuntimeException;
import com.walker.infrastructure.utils.JsonUtils;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.jdbc.BaseMapper;
import com.walker.jdbc.BasePo;
import com.walker.jdbc.Constants;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class JdbcUtils {

    private static final Logger logger = LoggerFactory.getLogger(JdbcUtils.class);

    public static final String MAPPER_SUFFIX = "_mapper";

    private static final Map<String, Class<?>> cachedMapperClass = new ConcurrentHashMap<>();

    /**
     * 根据输入PO对象，生成单表查询条件组装结果。
     * @param po
     * @return Object[0] = where.toString(), Object[1] = fieldNameAndValue（查询条件字段）
     * @param <T>
     * @date 2023-03-24
     */
    public static final <T extends BasePo<T>> Object[] acquireQueryCondition(T po){
        Object[] result = new Object[2];

        // po转成json对象，获取里面所有存在值的字段信息
        ObjectNode objectNode = JsonUtils.javaObjectToObjectNode(po);

        Map.Entry<String, JsonNode> entry = null;
        String fieldName = null;
        String valueText = null;
        JsonNode jsonNode = null;
        // 字段名、字段值集合
        Map<String, Object> fieldNameAndValue = new HashMap<>(4);
        StringBuilder where = new StringBuilder(Constants.SQL_NAME_WHERE);
        int i = 0;
        for(Iterator<Map.Entry<String, JsonNode>> it = objectNode.fields(); it.hasNext();){
            entry = it.next();
            if(entry.getKey().equals(Constants.PO_FIELD_PARAMETER)){
                // 额外的参数不属于表字段，剔除
                continue;
            }

            jsonNode = entry.getValue();

            // 2023-03-24 注意：这里jsonNode可能为 "null" 字符串
            if(jsonNode == null || jsonNode.toString().equals(com.walker.infrastructure.utils.StringUtils.NULL_STRING)){
//                logger.debug("........entry.getValue() = null");
                continue;
            }

            // 字段名字，转成下划线数据库字段名
            fieldName = com.walker.infrastructure.utils.StringUtils.transferCamelName2Underline(entry.getKey());

//            logger.debug("{}", jsonNode);
            if(i > 0){
                where.append(com.walker.infrastructure.utils.StringUtils.CHAR_SPACE).append(Constants.SQL_NAME_AND);
            }

            if(jsonNode.isTextual()){
                valueText = jsonNode.asText();
                logger.debug("valueText = {}", valueText);
                if(com.walker.infrastructure.utils.StringUtils.isEmpty(valueText)){
                    valueText = com.walker.infrastructure.utils.StringUtils.EMPTY_STRING;
                }
                if(valueText.indexOf(com.walker.infrastructure.utils.StringUtils.CHAR_PERCENT) >= 0){
                    // like 查询
                    where.append(com.walker.infrastructure.utils.StringUtils.CHAR_SPACE).append(fieldName).append(com.walker.infrastructure.utils.StringUtils.CHAR_SPACE)
                            .append(Constants.SQL_NAME_LIKE).append(com.walker.infrastructure.utils.StringUtils.CHAR_SPACE).append(com.walker.infrastructure.utils.StringUtils.SEPARATOR_COLON).append(fieldName);
                } else {
                    // 普通字符串查询
                    where.append(com.walker.infrastructure.utils.StringUtils.CHAR_SPACE).append(fieldName)
                            .append(com.walker.infrastructure.utils.StringUtils.CHAR_EQUALS).append(com.walker.infrastructure.utils.StringUtils.SEPARATOR_COLON).append(fieldName);
                }
                // 字符串会带双引号，奇怪！2023-03-24
                valueText = valueText.replace(com.walker.infrastructure.utils.StringUtils.STRING_DOUBLE_MARK, com.walker.infrastructure.utils.StringUtils.EMPTY_STRING);
                fieldNameAndValue.put(fieldName, valueText);

            } else {
                where.append(com.walker.infrastructure.utils.StringUtils.CHAR_SPACE).append(fieldName)
                        .append(com.walker.infrastructure.utils.StringUtils.CHAR_EQUALS).append(StringUtils.SEPARATOR_COLON).append(fieldName);
            }

            if(jsonNode.isLong()){
                fieldNameAndValue.put(fieldName, jsonNode.longValue());
            } else if(jsonNode.isDouble()){
                fieldNameAndValue.put(fieldName, jsonNode.doubleValue());
            } else if(jsonNode.isInt()){
                fieldNameAndValue.put(fieldName, jsonNode.intValue());
            } else if(jsonNode.isFloat()){
                fieldNameAndValue.put(fieldName, jsonNode.floatValue());
            } else {
                fieldNameAndValue.put(fieldName, jsonNode.asText());
            }
            i++;
        }
        if(where.length() <= 5){
            logger.warn("po未设置任何查询字段:{}", po.getClass().getName());
            where.append(" 1=1 ");
        }
        result[0] = where.toString();
        result[1] = fieldNameAndValue;
        return result;
    }

    /**
     * 生成给定PO的一个实例。
     * @param basePo
     * @return
     * @param <T>
     * @date 2023-09-20
     */
    public static <T extends BasePo<T>> BaseMapper<T> getInstance(T basePo) {
        try {
//            Class mapperClass = Class.forName("com.walker.model.po" + basePo.getClass().getSimpleName() + "_mapper");
            String className = basePo.getClass().getName() + MAPPER_SUFFIX;
            Class<?> mapperClass = cachedMapperClass.get(className);
            if(mapperClass == null){
                mapperClass = Class.forName(className);
                cachedMapperClass.put(className, mapperClass);
            }
            return (BaseMapper<T>) BeanUtils.instantiateClass(mapperClass.getConstructor(basePo.getClass()), basePo);
        } catch (Exception e) {
            throw new ApplicationRuntimeException(e.getMessage(), e);
        }
    }
}
