package com.walker.jdbc.util;

import com.walker.infrastructure.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TextUtils {

    protected static final transient Logger logger = LoggerFactory.getLogger(TextUtils.class);

    public static final char COLON_EN = ':';
    public static final char FOLDER_SEPARATOR = '/';

    /**
     * 根据jdbcUrl解析出来: ip | port | dbName。
     * <pre>
     *     1) jdbc:mysql://193.193.193.236:3306/cpmsxc?serverTimezone=Asia/Shanghai
     *     2) jdbc:oracle:thin:@61.163.210.202:6021/orcl
     *     3) jdbc:dm://192.168.100.100:5236?SCHEMA=iplatform
     *
     * </pre>
     * @param jdbcUrl
     * @return 如: ip = 193.193.193.236, port = 3306, dbName = cpmsxc
     */
    public static final String[] parseIpAndPortFromUrl(String jdbcUrl){
        boolean ipStart = false;
        boolean portStart = false;
        boolean dbStart = false;
        StringBuilder ipValue = null;
        StringBuilder portValue = null;
        StringBuilder dbValue = null;
        for(char c : jdbcUrl.toCharArray()){
            if(c == 32){
                // 空格不管
                continue;
            }
            if(Character.isDigit(c)){
                // 碰到的第一个数字，表示ip开始解析
                if(!ipStart && !portStart){
                    ipValue = new StringBuilder();
                    ipStart = true;
                }
                if(ipStart){
                    ipValue.append(c);
                    continue;
                }
                if(portStart){
                    portValue.append(c);
                    continue;
                }
            } else if(c == StringUtils.EXTENSION_SEPARATOR){
                // 如果处在ip解析过程，则作为ip一部分
                if(ipStart){
                    ipValue.append(c);
                    continue;
                }
            } else if(c == COLON_EN && ipStart){
                // 结束ip，开始解析端口
                portValue = new StringBuilder();
                portStart = true;
                ipStart = false;
            } else if(c == FOLDER_SEPARATOR && portStart){
                // 遇到斜杠，并且端口开始解析时，表示端口结束(数据库名字开始)
                dbValue = new StringBuilder();
                portStart = false;
                dbStart = true;
            } else if(c == '?') {
                if(jdbcUrl.indexOf(DB_TYPE_DAMENG_PREFIX) >= 0){
                    // 2023-03-02
                    // 达梦数据库，没有独立的数据库名称，都在后面参数中，把参数当作数据库解析
                    dbValue = new StringBuilder();
                    portStart = false;
                    dbStart = true;
                } else {
                    // 其他数据库正常走
                    dbStart = false;
                    break;
                }
            } else if(dbStart){
                dbValue.append(c);
            }
        }

        // 达梦：dbValue中是所有参数，解析数据库名称
        if(jdbcUrl.indexOf(DB_TYPE_DAMENG_PREFIX) >= 0){
            String dbName = getDamengDbname(dbValue.toString());
            logger.debug("找到'达梦'数据库:{}", dbName);
            dbValue = new StringBuilder(dbName);
        }

        if(ipValue == null || portValue == null || dbValue == null){
            logger.error("解析ip和port错误:ip或端口、数据库名称不存在, jdbcUrl = " + jdbcUrl);
            return null;
        }
        String[] ipAndPort = new String[3];
        ipAndPort[0] = ipValue.toString();
        ipAndPort[1] = portValue.toString();
        ipAndPort[2] = dbValue.toString();
        return ipAndPort;
    }

    /**
     * 查找达梦url中的数据库名字。
     * @param dbValue 如：SCHEMA=SYSDBA&LOGINMODE=4
     * @return
     */
    private static final String getDamengDbname(String dbValue){
        String[] keyValues = dbValue.split("&");
        String[] parameter = null;
        for(String one : keyValues){
            parameter = one.split("=");
            if(parameter[0].equalsIgnoreCase("schema")){
                return parameter[1];
            }
        }
        return null;
    }

    public static final String DB_TYPE_DAMENG_PREFIX = "jdbc:dm:";
}
