package com.walker.openocr.table;

import com.walker.openocr.MultipleLine;
import com.walker.openocr.TextBlock;

/**
 * 表格中，一个单元格定义。
 * @author 时克英
 * @date 2022-08-31
 */
public class CellObject implements Comparable<CellObject> {

    // 单元格编号，如果是用户不关注的，则随机生成
    private String id;

    // 行号，对应配置中用户定义的：CellConfigItem#orderNum
    private int rowNum = 0;

    // 在同一行中，该单元格所处列，数值越大越靠后（排序使用）
    private int orderColumn = 0;

    // 单元格对应的原始文本块数据
    private TextBlock source = null;

    private CellConfigItem cellConfigItem = null;

    // 单元格解析的最终值
    private String value = null;

    public String getValue() {
        return value;
    }

    public void setValue(String value) {
        this.value = value;
    }

    public CellConfigItem getCellConfigItem() {
        return cellConfigItem;
    }

    public void setCellConfigItem(CellConfigItem cellConfigItem) {
        this.cellConfigItem = cellConfigItem;
    }

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    public int getRowNum() {
        return rowNum;
    }

    public void setRowNum(int rowNum) {
        this.rowNum = rowNum;
    }

    public int getOrderColumn() {
        return orderColumn;
    }

    public void setOrderColumn(int orderColumn) {
        this.orderColumn = orderColumn;
    }

    public TextBlock getSource() {
        return source;
    }

    public void setSource(TextBlock source) {
        this.source = source;
    }

    /**
     * 判断该单元格是否匹配到表格配置项中，如果匹配了说明是用户需要的关键单元格（标题）
     * @return
     * @date 2022-09-01
     */
    public boolean isConfigurable(){
        return this.cellConfigItem != null;
    }

    /**
     * 返回：是不是一个单元格，有些数据不在格子里，需要按照规则拆分标题和值。如：name=保险单号:PDFC2261010420000000068
     * @return
     */
    public boolean isConfigNoneCell(){
        this.checkConfigItem();
        return this.cellConfigItem.isNoneCell();
    }

    /**
     * 返回：是否完整一行单元格
     * @return
     */
    public boolean isConfigFullRow(){
        this.checkConfigItem();
        return this.cellConfigItem.isFullRow();
    }

    public boolean isConfigTwoLine(){
        this.checkConfigItem();
        return this.cellConfigItem.getMultipleLine() == MultipleLine.TwoLine;
    }
    public boolean isConfigMoreLine(){
        this.checkConfigItem();
        return this.cellConfigItem.getMultipleLine() == MultipleLine.More;
    }
    public boolean isConfigSingleLine(){
        this.checkConfigItem();
        return this.cellConfigItem.getMultipleLine() == MultipleLine.None;
    }

    /**
     * 返回配置的该单元格值，最小字符串数量。
     * @return
     */
    public int getMinValueSize(){
        this.checkConfigItem();
        return cellConfigItem.getMinValueSize();
    }

    private void checkConfigItem(){
        if(this.cellConfigItem == null){
            throw new IllegalArgumentException("单元格不是可配置元素:" + this.id);
        }
    }

    @Override
    public int hashCode(){
        return this.id.hashCode();
    }

    @Override
    public boolean equals(Object obj){
        if(obj == null){
            return false;
        }
        if(obj instanceof CellObject){
            CellObject cellObject = (CellObject) obj;
            if(cellObject.id.equals(this.id)){
                return true;
            }
        }
        return false;
    }

    @Override
    public String toString(){
        StringBuilder sb = new StringBuilder("[id=");
        sb.append(this.id);
        sb.append(", row=").append(this.rowNum).append(", column=").append(this.orderColumn);
        if(this.source != null){
            sb.append(", name=").append(this.source.getText());
        }
        if(this.value != null){
            sb.append(", value=").append(this.value);
        }
        sb.append("]");
        return sb.toString();
    }

//    @Override
//    public int compare(CellObject o1, CellObject o2) {
//        return o1.orderColumn - o2.orderColumn;
//    }

    @Override
    public int compareTo(CellObject o) {
        return this.orderColumn - o.orderColumn;
    }
}
