package com.walker.openocr.table;

import com.walker.infrastructure.utils.StringUtils;
import com.walker.openocr.Constants;
import com.walker.openocr.MultipleLine;
import com.walker.openocr.util.JsonUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.Map;

public abstract class AbstractTableConfigLoader implements TableConfigLoader{

    protected final transient Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final String NAME_TABLE_TITLE = "table_title";
    private static final String NAME_TABLE_TYPE_KEYS = "table_type_keys";
    private static final String NAME_TABLE_TYPE_KEYS_NOT = "table_type_keys_not";
    private static final String NAME_TABLE_SETTING = "table_setting";
    private static final String NAME_REMOVE_COLUMNS = "remove_columns";

    private static final String KEY_CELL_TOLERANCE = "cell_tolerance";
    private static final String KEY_TITLE_TOLERANCE = "title_tolerance";
    private static final String KEY_MULTI_LINE_TOLERANCE = "multi_line_tolerance";
    private static final String KEY_CELL_NAME = "name";
    private static final String KEY_MULTI_LINE = "multi_line";
    private static final String KEY_NONE_CESS = "none_cell";
    private static final String KEY_END_FLAG = "end_flag";
    private static final String KEY_ALIGN = "align";
    private static final String KEY_DATA_TYPE = "data_type";
    private static final String KEY_ORDER_NUM = "order_num";
    private static final String KEY_FULL_ROW = "full_row";
    private static final String KEY_MIN_VALUE_SIZE = "min_value_size";


    @Override
    public TableConfig load(Object option) {
        List<String> configContent = this.doLoadContent(option);
        if(configContent == null || configContent.size() == 0){
            logger.error("未加载到配置项内容:" + option);
            return null;
        }

        TableConfig tableConfig = new TableConfig();

        try {
            String[] keyValue = null;
            for (String line : configContent) {
                // 忽略注释行
                if (line.startsWith(Constants.VALUE_SHARP)) {
                    continue;
                }
                keyValue = line.split(Constants.EQUALS);
                if (keyValue.length != 2) {
//                    throw new IllegalArgumentException("解析表格配置行错误，等号左右必须都有值:" + line);
                    if(keyValue.length == 1){
                        logger.warn("该属性没有配置任何值:" + line);
                        continue;
                    }
                    throw new IllegalArgumentException("解析表格配置行错误，等号左右必须都有值，配置不正确:" + line);
                }

                logger.debug("加载key:" + keyValue[0] + ", value=" + keyValue[1]);

                if (keyValue[0].equals(NAME_TABLE_TYPE_KEYS)) {
                    this.processTableTypeKeys(keyValue[1], tableConfig);

                } else if(keyValue[0].equals(NAME_TABLE_TYPE_KEYS_NOT)){
                    // 2023-11-01 添加表格标题可排除的关键词
                    this.processTableTypeKeysNot(keyValue[1], tableConfig);

                } else if (keyValue[0].equals(NAME_TABLE_SETTING)) {
                    this.processTableSetting(keyValue[1], tableConfig);

                } else if (keyValue[0].equals(NAME_REMOVE_COLUMNS)) {
                    this.processRemoveColumns(keyValue[1], tableConfig);

                } else if (keyValue[0].equals(NAME_TABLE_TITLE)) {
                    tableConfig.setTableTitle(keyValue[1]);

                } else {
                    this.processCellConfig(keyValue[0], keyValue[1], tableConfig);
                }
            }
            logger.info("加载完成 TableConfig:" + option);
        } catch (Exception ex){
            logger.error("解析表格配置出现错误:" + option, ex);
        }
        return tableConfig;
    }

    private void processCellConfig(String key, String value, TableConfig tableConfig) throws Exception{
        Map<String, Object> map = JsonUtils.jsonStringToObject(value, Map.class);
        if(map == null || map.size() == 0){
            return;
        }
        CellConfigItem item = new CellConfigItem();
        item.setId(key);

        Object name = map.get(KEY_CELL_NAME);
        if(name != null){
            item.setName(name.toString());
        }
        Object orderNum = map.get(KEY_ORDER_NUM);
        if(orderNum != null){
            item.setOrderNum(Integer.parseInt(orderNum.toString()));
        }
        Object align = map.get(KEY_ALIGN);
        if(align != null){
            item.setAlign(align.toString());
        }
        Object multipleLine = map.get(KEY_MULTI_LINE);
        if(multipleLine != null){
            item.setMultipleLine(MultipleLine.getType(multipleLine.toString()));
        }
        Object dataType = map.get(KEY_DATA_TYPE);
        if(dataType != null){
            item.setDataType(dataType.toString());
        }
        Object noneCell = map.get(KEY_NONE_CESS);
        if(noneCell != null){
            item.setNoneCell(Boolean.parseBoolean(noneCell.toString()));
        }
        Object endFlag = map.get(KEY_END_FLAG);
        if(endFlag != null){
            item.setEndFlag(endFlag.toString());
        }
        Object fullRow = map.get(KEY_FULL_ROW);
        if(fullRow != null){
            item.setFullRow(Boolean.parseBoolean(fullRow.toString()));
        }
        Object minValueSize = map.get(KEY_MIN_VALUE_SIZE);
        if(minValueSize != null){
            item.setMinValueSize(Integer.parseInt(minValueSize.toString()));
        }
        tableConfig.addCellConfig(item);
    }

    private void processRemoveColumns(String text, TableConfig tableConfig){
        String[] names = text.split(Constants.VALUE_SPLIT);
        tableConfig.setRemoveColumnsName(names);
    }

    private void processTableSetting(String text, TableConfig tableConfig) throws Exception{
        Map<String, Object> map = JsonUtils.jsonStringToObject(text, Map.class);
        if(map == null || map.size() == 0){
            throw new IllegalArgumentException("未找到表格配置项:" + NAME_TABLE_SETTING);
        }
        Object cellTolerance = map.get(KEY_CELL_TOLERANCE);
        if(cellTolerance != null){
            tableConfig.setCellTolerance(Integer.parseInt(cellTolerance.toString()));
        }
        Object titleTolerance = map.get(KEY_TITLE_TOLERANCE);
        if(titleTolerance != null){
            tableConfig.setTitleTolerance(Float.parseFloat(titleTolerance.toString()));
        }
        Object multiLineTolerance = map.get(KEY_MULTI_LINE_TOLERANCE);
        if(multiLineTolerance != null){
            tableConfig.setMultiLineTolerance(Integer.parseInt(multiLineTolerance.toString()));
        }
    }

    private void processTableTypeKeys(String text, TableConfig tableConfig){
        tableConfig.setTableTypeKeys(text);
    }

    private void processTableTypeKeysNot(String text, TableConfig tableConfig){
        if(StringUtils.isNotEmpty(text)){
            List<String> list = StringUtils.asList(text.split(StringUtils.DEFAULT_SPLIT_SEPARATOR));
            tableConfig.setTableTitleNotKey(list);
        }
    }

    protected abstract List<String> doLoadContent(Object option);
}
