package com.walker.pay;

import com.walker.infrastructure.arguments.Variable;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.pay.exception.NotifyException;
import com.walker.pay.exception.OrderException;
import com.walker.pay.response.OrderStatusResponsePay;
import com.walker.pay.util.PayDefinitionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public abstract class AbstractPayEngineManager implements PayEngineManager{

    protected final transient Logger logger = LoggerFactory.getLogger(this.getClass());

    private final Map<String, PayDefinition> payDefinitionMap = new HashMap<>();

    private final Map<String, PayEngineProvider> payEngineProviderMap = new HashMap<>();

    private Map<String, Variable> configuration = null;

    public Map<String, Variable> getConfiguration() {
        return configuration;
    }

    public void setConfiguration(Map<String, Variable> configuration){
        if(configuration == null || configuration.size() == 0){
            logger.warn("支付引擎配置参数为空，请确认是否不需要参数。");
        }
        this.configuration = configuration;
        logger.info("setConfiguration:{}", this.configuration);
    }

    @Override
    public void loadPayDefinitionList() {
        List<PayDefinition> list = this.acquirePayDefinitionList();
        if(StringUtils.isEmptyList(list)){
            logger.warn("未加载到任何'支付定义'集合, 无法使用支付功能!");
            return;
        }
        PayEngineProvider payEngineProvider = null;
        for(PayDefinition pd : list){
            try {
                if(!pd.getEnabled()){
                    logger.warn("支付定义被禁用，无法使用:" + pd.getName());
                    continue;
                }
                payEngineProvider = this.acquirePayEngineProvider(pd);
                if(payEngineProvider == null){
                    logger.error("acquirePayEngineProvider()得到空的引擎对象!");
                    continue;
                }
                this.payEngineProviderMap.put(pd.getId(), payEngineProvider);
                this.payDefinitionMap.put(pd.getId(), pd);
                logger.info("加载了一个'PayEngineProvider': " + pd.getName() + ", id=" + pd.getId());

            } catch (PayEngineNotFoundException e) {
                logger.error("获取支付引擎错误，未找到需要的对象: " + pd.getId());
                throw new RuntimeException(e);
            }
        }
    }

    @Override
    public PayDefinition getPayDefinition(String payDefinitionId) {
        PayDefinition pd = this.payDefinitionMap.get(payDefinitionId);
        if(pd == null){
            logger.warn("未找到支付定义对象, id=" + payDefinitionId);
        }
        return pd;
    }

    @Override
    public List<PayEngineProvider> getPayEngineProviderList() {
        if(this.payEngineProviderMap.size() == 0){
            return null;
        }
        List<PayEngineProvider> resultList = new ArrayList<>(8);
        for(PayEngineProvider payEngineProvider : this.payEngineProviderMap.values()){
            resultList.add(payEngineProvider);
        }
        return resultList;
    }

    @Override
    public PayEngineProvider getPayEngineProvider(String payDefinitionId) {
        PayEngineProvider payEngineProvider = this.payEngineProviderMap.get(payDefinitionId);
        if(payEngineProvider == null){
            logger.warn("未找到支付引擎对象, id=" + payDefinitionId);
        }
        return payEngineProvider;
    }

    @Override
    public PayEngineProvider getPayEngineProvider(ServiceProvider serviceProvider, String version, PayType payType) {
        String payDefinitionId = PayDefinitionUtils.getPayDefinitionId(serviceProvider, version);
        PayEngineProvider payEngineProvider = this.payEngineProviderMap.get(payDefinitionId);
        if(payEngineProvider == null){
            logger.warn("未找到支付引擎对象, id=" + payDefinitionId + ", provider=" + serviceProvider.getName() + ", version=" + version);
        }
        return payEngineProvider;
    }

    @Override
    public List<PayEngineProvider> getPayEngineProviderList(ServiceProvider serviceProvider){
        if(serviceProvider == null){
            throw new IllegalArgumentException("serviceProvider 参数必须提供");
        }
        if(this.payEngineProviderMap.size() == 0){
            return null;
        }
        List<PayEngineProvider> resultList = new ArrayList<>(2);
        for(PayEngineProvider payEngineProvider : this.payEngineProviderMap.values()){
            if(payEngineProvider.getServiceProvider() == serviceProvider){
                resultList.add(payEngineProvider);
            }
        }
        return resultList;
    }

    @Override
    public ResponsePay generatePrepareOrder(Order platformOrder) throws OrderException {
        if(platformOrder == null){
            throw new IllegalArgumentException("platformOrder is required!");
        }

        if(platformOrder.getServiceProvider() == null){
            throw new IllegalArgumentException("提供商不能为空: serviceProvider is required!");
        }
        if(platformOrder.getPayType() == null){
            throw new IllegalArgumentException("支付类型不能为空: payType is required!");
        }

        PayEngineProvider payEngineProvider = null;

        if(StringUtils.isEmpty(platformOrder.getVersion())){
            logger.warn("订单未传入'version'属性, 需要根据'ServiceProvider'查询是否存在多个提供者:" + platformOrder.getServiceProvider());
            List<PayEngineProvider> payEngineProviders = this.getPayEngineProviderList(platformOrder.getServiceProvider());
            if(StringUtils.isEmptyList(payEngineProviders)){
                logger.error("根据'serviceProvider'没有找到任何支付引擎提供者:" + platformOrder.getServiceProvider());
            } else if(payEngineProviders.size() == 1){
                if(logger.isDebugEnabled()){
                    logger.debug("该'serviceProvider'下只存在一个提供者:" + platformOrder.getServiceProvider());
                }
                payEngineProvider = payEngineProviders.get(0);
            } else {
                throw new IllegalArgumentException("订单没有提供版本号，而且该提供者存在多个，请传入版本号，serviceProvider=" + platformOrder.getServiceProvider());
            }
        } else {
            payEngineProvider = this.getPayEngineProvider(platformOrder.getServiceProvider()
                    , platformOrder.getVersion(), platformOrder.getPayType());
        }
        if(payEngineProvider == null){
            throw new IllegalArgumentException("PayEngineProvider not found:" + platformOrder.getServiceProvider() + ", " + platformOrder.getVersion());
        }
        return payEngineProvider.generatePrepareOrder(platformOrder);
    }

    @Override
    public String notifyOrder(String payDefinitionId, Object notifyData) throws NotifyException{
        if(StringUtils.isEmpty(payDefinitionId)){
//            throw new NotifyException("", "payDefinitionId 为空!", null);
            return "payDefinitionId 为空!";
        }

        PayEngineProvider payEngineProvider = this.getPayEngineProvider(payDefinitionId);
        if(payEngineProvider == null){
//            throw new IllegalArgumentException("未找到支付引擎，payDefinition:" + payDefinitionId);
            return "未找到支付引擎，payDefinition:" + payDefinitionId;
        }

//        payEngineProvider.notifyOrder(notifyData, orderCallBack);
        payEngineProvider.notifyOrder(notifyData);

        // 成功返回空对象。
        return null;
    }

    @Override
    public String generateNotifyResponse(String payDefinitionId, boolean success){
        if(StringUtils.isEmpty(payDefinitionId)){
            throw new IllegalArgumentException("payDefinitionId is acquired!");
        }
        PayEngineProvider payEngineProvider = this.getPayEngineProvider(payDefinitionId);
        if(payEngineProvider == null){
            throw new IllegalArgumentException("未找到支付引擎，payDefinition:" + payDefinitionId);
        }
        return payEngineProvider.generateNotifyResponse(success);
    }

    @Override
    public OrderStatusResponsePay searchOrderStatus(String orderId){
        if(StringUtils.isEmpty(orderId)){
            throw new IllegalArgumentException("请输入订单编号");
        }
        Order order = this.acquireOrder(orderId);
        if(order == null){
            logger.error("系统订单不存在，orderId={}", orderId);
            return null;
        }
        String payDefinitionId = order.getPayDefinitionId();
        if(StringUtils.isEmpty(payDefinitionId)){
            throw new IllegalStateException("该订单未关联任何支付引擎，无法继续查询订单状态: payDefinitionId 为空，orderId=" + orderId);
        }
        PayEngineProvider payEngineProvider = this.getPayEngineProvider(payDefinitionId);
        if(payEngineProvider == null){
            throw new IllegalArgumentException("未找到支付引擎，payDefinition:" + payDefinitionId);
        }
        return payEngineProvider.searchOrderStatus(order);
    }

    /**
     * 由子类实现加载支付定义数据。
     * @return
     */
    protected abstract List<PayDefinition> acquirePayDefinitionList();

    /**
     * 根据支付定义，获取支付引擎提供者对象。
     * @param payDefinition
     * @return
     */
    protected abstract PayEngineProvider acquirePayEngineProvider(PayDefinition payDefinition) throws PayEngineNotFoundException;

    /**
     * 从平台查询数据库，找出订单基本信息。
     * @param orderId 平台订单ID，数字
     * @return
     * @date 2023-02-23
     */
    protected abstract Order acquireOrder(String orderId);
}
