package com.walker.pay;

import java.util.List;

/**
 * 订单接口，是对支付订单的抽象。<p></p>
 * 不同第三方都会使用该规范，业务可以根据需要继承该规范添加自己的属性。
 * @author 时克英
 * @date 2023-01-16
 */
public interface Order {

    /**
     * 返回系统订单号
     * @return
     */
    long getId();

    /**
     * 返回系统订单号（字符串），如果id是数值则会自动转换成字符串。<p></p>
     * 该方法与getId()同时只能使用一种。
     * @return
     * @date 2023-03-02
     */
    String getIdString();

    /**
     * 返回支付定义ID，该方法值由平台使用，业务暂时获取不到。<p></p>
     * 因为业务肯定不知道 '支付定义ID'。
     * @return
     * @date 2023-01-17 更新
     * @date 2023-02-23 该参数在查询订单状态时需要使用，由平台自动设置
     */
    String getPayDefinitionId();

    /**
     * 返回订单使用支付提供商。
     * @return
     * @date 2023-01-17
     */
    ServiceProvider getServiceProvider();

    /**
     * 返回支付提供者实现的版本号。如: v2
     * @return
     * @date 2023-01-17
     */
    String getVersion();

    /**
     * 返回第三方支付提供商生成的订单编号,如:微信订单号、支付宝订单号
     * @return
     */
    String getTradeNo();

    /**
     * 返回支付类型，参见枚举 {@linkplain PayType}
     * @return
     */
    PayType getPayType();

    /**
     * 返回实际支付总金额，单位: 分
     * @return
     */
    long getTotalMoney();

    /**
     * 返回商品标题(描述)
     * @return
     */
    String getTitle();

    /**
     * 得到一个随机数。
     * @return
     */
    String getNonce();

    /**
     * 返回订单通知回调地址(业务配置)
     * @return
     */
    String getNotifyUrl();

    /**
     * 返回业务传入的附加数据，该数据第三方一般会原封不动返回。
     * @return
     */
    String getAttach();

    /**
     * 返回终端ip地址。
     * @return
     */
    String getIp();

    long getCreateTime();

    /**
     * 返回用户提交的设备信息。
     * @return
     * @date 2023-02-17
     */
    String getDeviceInfo();

    /**
     * 返回订单手续费，内扣，如果不存在，则填0。
     * 单位：分。如amount为100，fee为2，实际到账金额为98，平台手续费收入为2。<p></p>
     * 该参数对接 '通商云' 时使用到。
     * @return
     * @date 2023-02-17
     */
    long getFee();

    /**
     * 返回 买家ID，'通商云' 时使用到
     * @return
     * @date 2023-02-17
     */
    String getBuyerId();

    /**
     * 返回 卖家ID，'通商云' 时使用到
     * @return
     * @date 2023-02-17
     */
    String getReceiverId();

    /**
     * 返回订单支付状态，在查询是需要从数据库加载出来。
     * @return
     * @date 2023-02-23
     */
    PayStatus getPayStatus();

    /**
     * 返回分账用户集合，支持多人分账，但目前不支持多层级分账。
     * @return
     * @author 时克英
     * @date 2023-04-03
     */
    List<SplitAccount> getSplitAccountList();
}
