package com.walker.pay;

import com.walker.infrastructure.arguments.Variable;
import com.walker.pay.callback.OrderCallBack;

import java.util.Map;

/**
 * 支付定义，产生一个支付自定义的实现，包括:支付基本信息、实现接口等。<p></p>
 * <pre>
 *     1)每个定义只能接入一家提供商，如:微信、通联等。
 *     2)每个定义都实现统一的支付能力，如:下单、通知、退款、对账等接口能力。
 * </pre>
 * @author 时克英
 * @date 2023-01-12
 */
public interface PayDefinition {

    /**
     * 支付定义ID，唯一，提供商标识 + 版本号
     * @return
     */
    String getId();

    /**
     * 支付定义名字，用户可录入，默认使用:提供商名称 + 版本号 + 支付渠道
     * @return
     */
    String getName();

    /**
     * 返回提供商对象
     * @return
     */
    ServiceProvider getServiceProvider();

    /**
     * 返回接入版本号
     * @return
     */
    String getVersion();

    /**
     * 返回支付渠道
     * @return
     */
    PayChannel getPayChannel();

    /**
     * 是否启用
     * @return
     */
    boolean getEnabled();

    /**
     * 返回支付配置参数集合
     * @return
     */
    Map<String, Variable> getConfiguration();

//    /**
//     * 返回支付引擎对象。
//     * @return
//     */
//    PayEngineProvider getPayEngineProvider();

    /**
     * 返回定义对应的支付引擎实现类路径, 如: com.walker.pay.wechat.WechatV2EngineProvider
     * @return
     * @date 2023-01-17
     */
    String getPayEngineProviderClass();

    /**
     * 返回支付定义配置的 '订单回调'操作实现对象。<p></p>
     * 引擎内部默认实现不做任何业务处理，业务集成后应当重新继承并实现。
     * @return
     * @date 2023-02-20
     */
    OrderCallBack getOrderCallback();

    /**
     * 返回订单通知数据格式转换器对象。
     * @return
     * @date 2023-02-26
     * @date 2023-03-01 由管理器（根据类路径）组装好对象，这里可以返回直接使用。<br>
     * 参考：{@linkplain PayDefinition#getOrderNotifyConvertorClass()}
     */
//    @Deprecated
    Convertor<NotifyValue<?>> getOrderNotifyConvertor();

    /**
     * 返回订单通知转换器类路径，用于动态创建，从数据库读取配置的类路径。
     * @return
     */
    String getOrderNotifyConvertorClass();
}
