package com.walker.pay;

import com.walker.pay.exception.NotifyException;
import com.walker.pay.exception.OrderException;
import com.walker.pay.response.OrderStatusResponsePay;

import java.util.List;

/**
 * 支付引擎管理器定义。
 * <p>业务系统使用该对象统一调用支付接口，完成支付、通知、退款等相关操作。</p>
 * <pre>
 *     1)该接口对外使用，内部会根据参数使用不同的具体支付提供商，转发支付请求。
 * </pre>
 * @author 时克英
 * @date 2023-01-17
 */
public interface PayEngineManager {

//    void register(PayEngineProvider payEngineProvider);

    /**
     * 加载业务配置的所有支付定义记录。
     * @return
     */
    void loadPayDefinitionList();

    /**
     * 根据支付配置ID，返回支付定义对象。
     * @param payDefinitionId
     * @return
     */
    PayDefinition getPayDefinition(String payDefinitionId);

    /**
     * 返回所有可用的支付引擎提供者集合。
     * @return
     */
    List<PayEngineProvider> getPayEngineProviderList();

    /**
     * 根据支付定义ID，返回具体的支付引擎对象。如: 微信支付提供者等。<p></p>

     * @param payDefinitionId
     * @return
     */
    PayEngineProvider getPayEngineProvider(String payDefinitionId);

    /**
     * 根据订单信息，返回具体的支付引擎对象。
     * <pre>
     *     1)该方法在支付通知时会被内部使用，来确定哪个提供者被调用。
     * </pre>
     * @param serviceProvider 支付提供者
     * @param version 支付提供引擎的版本号
     * @param payType (订单)支付类型
     * @return
     */
    PayEngineProvider getPayEngineProvider(ServiceProvider serviceProvider, String version, PayType payType);

    /**
     * 根据服务提供商信息，查询包含该提供商的支付对接有几个，因为可能存在多个，如:微信v2、v3等。<p></p>
     * 在订单没有传入version版本号的时候，可以检测如果提供商只有一个实现，则无需使用版本号。
     * @param serviceProvider
     * @return
     * @date 2023-01-17
     */
    List<PayEngineProvider> getPayEngineProviderList(ServiceProvider serviceProvider);

    /**
     * 生成预下单信息。
     * @param platformOrder
     */
    ResponsePay generatePrepareOrder(Order platformOrder) throws OrderException;

    /**
     * 订单(异步)通知调用方法。
     * @param payDefinitionId 支付定义ID，由业务确定，因为该方法必须放在独立http方法中被触发。
     * @param notifyData 通知原始数据
//     * @param orderCallBack 业务回调接口，可选
     * @return 如果返回不为空，则表示执行错误，返回了error字符串。
     * @throws NotifyException
     * @date 2023-02-17
     */
    String notifyOrder(String payDefinitionId, Object notifyData
//            , OrderCallBack orderCallBack
    ) throws NotifyException;

    /**
     * 生成接收通知后，给第三方的响应内容。
     * @param payDefinitionId 支付定义ID
     * @param success 是否成功
     * @return
     * @date 2023-02-17
     */
    String generateNotifyResponse(String payDefinitionId, boolean success);

    /**
     * 远程调用第三方接口，查询支付订单状态。
     * @param orderId 系统内部订单号
     * @return
     * @date 2023-02-23
     */
    OrderStatusResponsePay searchOrderStatus(String orderId);
}
