package com.walker.pay;

import com.walker.infrastructure.arguments.Variable;
import com.walker.pay.callback.OrderCallBack;
import com.walker.pay.exception.NotifyException;
import com.walker.pay.exception.OrderException;
import com.walker.pay.response.OrderStatusResponsePay;

import java.util.Map;

/**
 * 支付引擎提供者，抽象最基本的支付操作。
 * @author 时克英
 * @date 2023-01-12
 */
public interface PayEngineProvider {

    /**
     * 设置当前支付引擎的 '支付定义' 对象。
     * @param payDefinition
     * @date 2023-02-17
     */
    void setPayDefinition(PayDefinition payDefinition);

    /**
     * 设置当前支付引擎需要的配置参数。
     * @param configuration
     */
    void setConfiguration(Map<String, Variable> configuration);

    /**
     * 返回服务提供商信息。
     * @return
     */
    ServiceProvider getServiceProvider();

    /**
     * 返回该提供商支付对接的版本号。<p></p>
     * 例如:微信支付存在多个版本，因此服务商+版本号 确定唯一的提供者实现。
     * @return
     */
    String getVersion();

    /**
     * 返回渠道信息
     * @return
     */
    PayChannel getPayChannel();

    /**
     * 返回支付定义ID，应该是支付提供者来确定该定义ID，然后与定义集合对比。
     * @return
     * @date 2023-01-17
     */
    String getPayDefinitionId();

    /**
     * 返回第三方支付类型字符串，如: app/jsapi/native等，这些是第三方支付类型区别。
     * @return
     */
    String getProviderPayType(PayType payType);

    /**
     * 生成预下单的订单对象。
     * @param platformOrder
     * @return ResponseValue 返回调用结果
     * @throws OrderException 订单获取异常
     * @date 2023-01-16 更新返回结果
     */
    ResponsePay generatePrepareOrder(Order platformOrder) throws OrderException;

    /**
     * 查询支付订单状态。
     * @param order 系统内部订单对象
     * @return
     * @date 2023-02-23
     * @date 2023-08-30 输入参数改为订单对象
     */
//    OrderStatusResponsePay searchOrderStatus(String orderId);
    OrderStatusResponsePay searchOrderStatus(Order order);

    /**
     * 订单结果通知，业务在接收到第三方通知后，需要调用该方法处理具体流程。
     * @param notifyData 通知消息
//     * @param callBack 业务回调函数
     * @throws NotifyException
     * @date 2023-01-16
     * @date 2023-02-20 把 OrderCallback 放到引擎中，统一在引擎创建时配置。
     */
    void notifyOrder(Object notifyData
//            , OrderCallBack callBack
    ) throws NotifyException;

    /**
     * 生成接收通知后，给第三方的响应内容。
//     * @param payDefinitionId 支付定义ID
     * @param success 是否成功
     * @return
     * @date 2023-02-17
     */
    String generateNotifyResponse(boolean success);

    /**
     * 设置订单操作的回调实现，该接口定义了支付订单过程的一个关键节点的入口回调，<br>
     * 业务通过实现接口，避免支付与业务的耦合，起到分离作用。
     * @param orderCallBack
     * @date 2023-02-20
     */
    void setOrderCallback(OrderCallBack orderCallBack);
}
