package com.walker.pay.payunk;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.walker.infrastructure.ApplicationRuntimeException;
import com.walker.infrastructure.arguments.Variable;
import com.walker.infrastructure.utils.JsonUtils;
import com.walker.pay.Order;
import com.walker.pay.OrderGenerator;
import com.walker.pay.OrderStatusQuery;
import com.walker.pay.PayChannel;
import com.walker.pay.PayContext;
import com.walker.pay.PayStatus;
import com.walker.pay.PayType;
import com.walker.pay.ServiceProvider;
import com.walker.pay.payunk.generator.Behalf2AlipayOneGenerator;
import com.walker.pay.payunk.generator.Trans2AlipayOneGenerator;
import com.walker.pay.payunk.util.SignUtils;
import com.walker.pay.response.OrderStatusResponsePay;
import com.walker.pay.support.SimplePayEngineProvider;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

import java.util.HashMap;
import java.util.Map;

/**
 * 畅联（第三方）支付引擎实现。
 * @author 时克英
 * @date 2023-10-25
 */
public class PayUnkEngineProvider extends SimplePayEngineProvider {

    private Map<String, OrderGenerator> orderGeneratorMap = new HashMap<>(8);

    public PayUnkEngineProvider(RestTemplate restTemplate){
        if(restTemplate == null){
            throw new IllegalArgumentException("RestTemplate is required!");
        }
        this.restTemplate = restTemplate;
        this.init();
    }

    private void init(){
        this.setServiceProvider(ServiceProvider.PayUnk);
        this.setVersion(com.walker.pay.Constants.DEFAULT_VERSION);
        this.setPayChannel(PayChannel.ProviderDirect);

        Trans2AlipayOneGenerator trans2AlipayOneGenerator = new Trans2AlipayOneGenerator(this.restTemplate);
        this.orderGeneratorMap.put(Constants.PAY_TYPE_TRANS_ALIPAY_ONE, trans2AlipayOneGenerator);

        Behalf2AlipayOneGenerator behalf2AlipayOneGenerator = new Behalf2AlipayOneGenerator(this.restTemplate);
        this.orderGeneratorMap.put(Constants.PAY_TYPE_BEHALF_ALIPAY_ONE, behalf2AlipayOneGenerator);
    }

    @Override
    protected String acquireProviderPayType(ServiceProvider serviceProvider, PayType payType, PayChannel payChannel) {
        if(payType == PayType.PayUnk_TRANS_ALIPAY_ONE){
            return Constants.PAY_TYPE_TRANS_ALIPAY_ONE;
        } else if(payType == PayType.PayUnk_BEHALF_ALIPAY_ONE){
            return Constants.PAY_TYPE_BEHALF_ALIPAY_ONE;
        } else {
            throw new UnsupportedOperationException("代码未实现'畅联'支付类型转换:" + payType.getName());
        }
    }

    @Override
    protected OrderGenerator acquireOrderGenerator(String providerPayType, Order platformOrder, Map<String, Variable> configuration) {
        return this.orderGeneratorMap.get(providerPayType);
    }

    @Override
    protected boolean verifySign(Object notifyData) throws Exception {
        return true;
    }

    @Override
    protected PayContext acquirePayContext(String providerPayType, Order platformOrder, Map<String, Variable> configuration) {
        DefaultPayContext payContext = new DefaultPayContext();
        payContext.setConfigVariable(configuration.get(Constants.CONFIG_KEY_APPID));
        payContext.setConfigVariable(configuration.get(Constants.CONFIG_KEY_USER_ACCOUNT_ID));
//        payContext.setConfigVariable(configuration.get(Constants.CONFIG_KEY_APP_CERT_PATH));
//        payContext.setConfigVariable(configuration.get(Constants.CONFIG_KEY_ALIPAY_ROOT_CERT_PATH));
//        payContext.setConfigVariable(configuration.get(Constants.CONFIG_KEY_ALIPAY_APPID));
//        payContext.setConfigVariable(configuration.get(Constants.CONFIG_KEY_ALIPAY_CERT_PATH));
//        payContext.setConfigVariable(configuration.get(Constants.CONFIG_KEY_ALIPAY_RSA_PRIVATE_KEY));
        payContext.setConfigVariable(configuration.get(Constants.CONFIG_KEY_ALIPAY_IDENTITY));
        payContext.setConfigVariable(configuration.get(Constants.CONFIG_KEY_ALIPAY_AGREEMENT_NO));
        return payContext;
    }

    @Override
    protected OrderStatusQuery acquireOrderStatusQuery(Order order) {
        OrderStatusQuery query = new OrderStatusQuery();
        query.setTradeNo(String.valueOf(order.getId()));
        query.setAppId(this.getConfiguration().get(Constants.CONFIG_KEY_APPID).getStringValue());
        query.setMerchantId(this.getConfiguration().get(Constants.CONFIG_KEY_USER_ACCOUNT_ID).getStringValue());
        return query;
    }

    @Override
    protected OrderStatusResponsePay invokeOrderStatus(OrderStatusQuery orderStatusQuery) {
        Map<String, Object> payMap = new HashMap<>(10);
        payMap.put("appid", orderStatusQuery.getAppId());
        payMap.put("user_account_id", orderStatusQuery.getMerchantId());
        payMap.put("out_trade_no", orderStatusQuery.getTradeNo());  // 业务订单ID
        String signStr = SignUtils.signature(payMap);
        payMap.put("sign", signStr);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity<Map<String, Object>> request = new HttpEntity<>(payMap, headers);
        ResponseEntity<String> entity = this.restTemplate.postForEntity(Constants.URL_GET_ORDER_STATUS, request, String.class);

        OrderStatusResponsePay responsePay = new OrderStatusResponsePay();

        /**
         * {
         *     "code": 200,
         *     "msg": "获取订单成功",
         *     "data": {
         *         "order_no": "EA227904647657092233",
         *         "out_trade_no": "2021102213040658828123456",
         *         "pay_trade_no": "0",
         *         "appid": "APP202110130500445511023765",
         *         "sub_appid": "",
         *         "user_account_id": 2838,
         *         "money": "0.01",
         *         "status": 1,
         *         "pay_time": 0,
         *         "callback_status": 1
         *     }
         * }
         */
        String result = entity.getBody();
        try {
            ObjectNode objectNode = JsonUtils.jsonStringToObjectNode(result);
            if(objectNode.get("code").intValue() != 200){
                String error = objectNode.get("msg").toString();
                logger.error("支付宝个人提现订单查询错误：{}", error);
                responsePay.setStatus(false);
                responsePay.setMessage(error);

            } else {
                responsePay.setStatus(true);
                JsonNode data = objectNode.get("data");
                int payStatus = data.get("status").intValue();
                if(payStatus == 2 || payStatus == 1){
                    responsePay.setPayStatus(PayStatus.NotPay);
                    responsePay.setMessage("未支付");

                } else if(payStatus == 3){
                    // 订单超时
                    responsePay.setPayStatus(PayStatus.Closed);
                    responsePay.setMessage("已关闭");

                } else if(payStatus == 5){
                    responsePay.setPayStatus(PayStatus.Refund);
                    responsePay.setMessage("已退款");

                } else if(payStatus == 4){
                    responsePay.setAppId(data.get("appid").toString());
                    responsePay.setOrderId(data.get("out_trade_no").toString());// 商户（业务系统）订单号
                    responsePay.setTradeNo(data.get("order_no").toString());    // 畅联系统订单号
                    responsePay.setPayStatus(PayStatus.Success);
                    responsePay.setPaySuccessTime(data.get("pay_time").toString());
                    responsePay.setMerchantId(data.get("user_account_id").toString());
                    double moneyFen = data.get("amount").doubleValue() * 100;
                    responsePay.setTotalMoney((long)moneyFen);

                } else {
                    throw new UnsupportedOperationException("不支持支付宝（订单查询）返回的状态类型：" + payStatus);
                }
            }

        } catch (Exception e) {
            throw new ApplicationRuntimeException("转换'查询支付宝转账订单'错误：" + e.getMessage() + ", " + result, e);
        }
        return responsePay;
    }

    @Override
    public String generateNotifyResponse(boolean success) {
        if(success){
            return Constants.CALL_BACK_SUCCESS;
        } else {
            return Constants.CALL_BACK_FAILED;
        }
    }

    public void setRestTemplate(RestTemplate restTemplate) {
        this.restTemplate = restTemplate;
    }

    private RestTemplate restTemplate;
}
