package com.walker.pay.payunk.generator;

import com.fasterxml.jackson.databind.node.ObjectNode;
import com.walker.infrastructure.ApplicationRuntimeException;
import com.walker.infrastructure.arguments.Variable;
import com.walker.infrastructure.utils.JsonUtils;
import com.walker.infrastructure.utils.MoneyUtils;
import com.walker.pay.Order;
import com.walker.pay.PayContext;
import com.walker.pay.ResponsePay;
import com.walker.pay.payunk.BaseOrderGenerator;
import com.walker.pay.payunk.Constants;
import com.walker.pay.payunk.DefaultPayContext;
import com.walker.pay.payunk.Trans2AlipayResponsePay;
import com.walker.pay.payunk.util.AlipayUtils;
import com.walker.pay.payunk.util.SignUtils;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

import java.util.HashMap;
import java.util.Map;

/**
 * 转账到支付宝个人账户，生成器实现。
 * @author 时克英
 * @date 2023-10-25
 */
public class Trans2AlipayOneGenerator extends BaseOrderGenerator {

    public Trans2AlipayOneGenerator(RestTemplate restTemplate){
        this.setRestTemplate(restTemplate);
    }

    @Override
    protected ResponsePay invoke(PayContext payContext, Order platformOrder) {
        try {
            DefaultPayContext context = (DefaultPayContext) payContext;
            // 请求封装参数
            Map<String, Object> payMap = new HashMap<>(10);
            // 您的商户唯一标识
            payMap.put("appid", context.getAppIdConfig().getStringValue());
            // 支付通道
            payMap.put("pay_type", Constants.PAY_TYPE_TRANS_ALIPAY_ONE);
            // 畅联后台 账户管理 账户编号
            payMap.put("user_account_id", context.getUserAccountId().getStringValue());
            // 交易金额 单位：元 精确小数点后2位10.00
            payMap.put("amount", MoneyUtils.scaleYuan2Accuracy(platformOrder.getTotalMoney()));
            // 回调地址
            payMap.put("callback_url", platformOrder.getNotifyUrl());
            // 商户订单号	C20142222231234
            payMap.put("out_trade_no", String.valueOf(platformOrder.getId()));

            payMap.put("pay", JsonUtils.objectToJsonString(AlipayUtils.acquireAlipayReplace(context)));

            payMap.put("extend", AlipayUtils.getAlipayInfo(platformOrder));

            // sign	签名
            String signStr = SignUtils.signature(payMap);
            payMap.put("sign", signStr);

            //
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            HttpEntity<Map<String, Object>> request = new HttpEntity<>(payMap, headers);
            ResponseEntity<String> entity = this.getRestTemplate().postForEntity(PAY_RETURN_JSON_URL, request, String.class);

            Trans2AlipayResponsePay responsePay = new Trans2AlipayResponsePay();

            /**
             * 返回示例
             * {
             *     "code": 200,
             *     "msg": "获取成功",
             *     "url":	"支付连接",
             * }
             */
            String result = entity.getBody();
            ObjectNode objectNode = JsonUtils.jsonStringToObjectNode(result);
            if(objectNode.get("code").intValue() != 200){
                String error = objectNode.get("msg").toString();
                logger.error("支付宝个人提现接口错误：{}", error);
                responsePay.setStatus(false);
                responsePay.setMessage(error);
            } else {
                responsePay.setStatus(true);
                responsePay.setMessage("success");
                responsePay.setUrl(objectNode.get("url").toString());
                logger.error("支付宝个人提现接口成功：{}", responsePay.getUrl());
            }
            return responsePay;

        } catch (Exception ex){
            logger.error("调用支付接口异常，Trans2AlipayOneGenerator：" + ex.getMessage(), ex);
            throw new ApplicationRuntimeException("Trans2AlipayOneGenerator", ex);
        }
    }

    @Deprecated
    @Override
    protected ResponsePay invoke(String providerPayType, Order platformOrder, Map<String, Variable> configuration) {
        throw new UnsupportedOperationException("方法废弃，请勿调用");
    }
}
