package com.walker.support.milvus;

//import io.milvus.param.ParamUtils;

import org.apache.commons.lang3.StringUtils;

import java.util.HashMap;
import java.util.Map;

public class FieldType {
//    private FieldType fieldType = null;

    private String name;
    private boolean primaryKey;
    private String description;
    private DataType dataType;
    private Map<String, String> typeParams;

    private FieldType(FieldType.Builder builder) {
        if (builder == null) {
            throw new NullPointerException("builder is marked non-null but is null");
        } else {
            this.name = builder.name;
            this.primaryKey = builder.primaryKey;
            this.description = builder.description;
            this.dataType = builder.dataType;
            this.typeParams = builder.typeParams;
//            this.autoID = builder.autoID;
        }
    }

    public static FieldType.Builder newBuilder() {
        return new FieldType.Builder();
    }

    @Override
    public String toString(){
        return new StringBuilder("[name=").append(this.name)
                .append(", isPrimary=").append(this.primaryKey)
                .append(", dataType=").append(this.dataType)
                .append(", summary=").append(this.description)
                .append("]").toString();
    }

    public String getName(){
        return this.name;
    }
    public boolean isPrimaryKey(){
        return this.primaryKey;
    }
    public String getDescription(){
        return this.description;
    }
    public DataType getDataType(){
        return this.dataType;
    }
    public Map<String, String> getTypeParams(){
        return this.typeParams;
    }

    public static final class Builder {
        private String name;
        private boolean primaryKey;
        private String description;
        private DataType dataType;
        private final Map<String, String> typeParams;
//        private boolean autoID;

        private Builder() {
            this.primaryKey = false;
            this.description = "";
            this.typeParams = new HashMap();
//            this.autoID = false;
        }

        public Builder withName(String name) {
            if (name == null) {
                throw new NullPointerException("name is marked non-null but is null");
            } else {
                this.name = name;
                return this;
            }
        }

        public Builder withPrimaryKey(boolean primaryKey) {
            this.primaryKey = primaryKey;
            return this;
        }

        public Builder withDescription(String description) {
            if (description == null) {
                throw new NullPointerException("description is marked non-null but is null");
            } else {
                this.description = description;
                return this;
            }
        }

        public Builder withDataType(DataType dataType) {
            if (dataType == null) {
                throw new NullPointerException("dataType is marked non-null but is null");
            } else {
                this.dataType = dataType;
                return this;
            }
        }

        public Builder addTypeParam(String key, String value) {
            if (key == null) {
                throw new NullPointerException("key is marked non-null but is null");
            } else if (value == null) {
                throw new NullPointerException("value is marked non-null but is null");
            } else {
                this.typeParams.put(key, value);
                return this;
            }
        }

        public Builder withTypeParams(Map<String, String> typeParams) {
            if (typeParams == null) {
                throw new NullPointerException("typeParams is marked non-null but is null");
            } else {
                Map var10001 = this.typeParams;
                typeParams.forEach(var10001::put);
                return this;
            }
        }

        public Builder withDimension(Integer dimension) {
            if (dimension == null) {
                throw new NullPointerException("dimension is marked non-null but is null");
            } else {
                this.typeParams.put("dim", dimension.toString());
                return this;
            }
        }

        public Builder withMaxLength(Integer maxLength) {
            if (maxLength == null) {
                throw new NullPointerException("maxLength is marked non-null but is null");
            } else {
                this.typeParams.put("max_length", maxLength.toString());
                return this;
            }
        }

//        public Builder withAutoID(boolean autoID) {
//            this.autoID = autoID;
//            return this;
//        }

        public FieldType build() throws ParamException {
//            ParamUtils.CheckNullEmptyString(this.name, "Field name");
            if (this.name == null || StringUtils.isBlank(this.name)) {
                throw new ParamException(name + " cannot be null or empty");
            }
            if (this.dataType != null && this.dataType != DataType.None) {
                if (this.dataType == DataType.String) {
                    throw new ParamException("String type is not supported, use VarChar instead");
                } else {
                    int len;
                    if (this.dataType == DataType.FloatVector || this.dataType == DataType.BinaryVector) {
                        if (!this.typeParams.containsKey("dim")) {
                            throw new ParamException("Vector field dimension must be specified");
                        }

                        try {
                            len = Integer.parseInt((String)this.typeParams.get("dim"));
                            if (len <= 0) {
                                throw new ParamException("Vector field dimension must be larger than zero");
                            }
                        } catch (NumberFormatException var3) {
                            throw new ParamException("Vector field dimension must be an integer number");
                        }
                    }

//                    if (this.dataType == io.milvus.grpc.DataType.VarChar) {
//                        if (!this.typeParams.containsKey("max_length")) {
//                            throw new ParamException("Varchar field max length must be specified");
//                        }
//
//                        try {
//                            len = Integer.parseInt((String)this.typeParams.get("max_length"));
//                            if (len <= 0) {
//                                throw new ParamException("Varchar field max length must be larger than zero");
//                            }
//                        } catch (NumberFormatException var2) {
//                            throw new ParamException("Varchar field max length must be an integer number");
//                        }
//                    }

                    return new FieldType(this);
                }
            } else {
                throw new ParamException("Field data type is illegal");
            }
        }
    }
}
