package com.walker.support.redis.cache;

import com.walker.cache.AbstractCacheProvider;
import com.walker.cache.Cache;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.support.redis.RedisHelper;

import java.util.Map;

/**
 * Redis方式缓存提供者实现
 * @author shikeying
 *
 * @param <T>
 */
public abstract class RedisCacheProvider<T> extends AbstractCacheProvider<T> {

	private RedisHelper redisHelper;

	public void setRedisHelper(RedisHelper redisHelper) {
		this.redisHelper = redisHelper;
	}

	@Override
	protected Cache provideCacheInstance(String name, Map<String, String> param){
//		RedisCache cache = null;
		if(this.isUseRedis()){
			RedisCache cache = new RedisCache(name, param);
			cache.setRedisHelper(this.redisHelper);
			cache.setSupportExpiredCache(this.supportExpiredCache);	// 2024-01-05
			return cache;
		} else {
			return super.provideCacheInstance(name, param);
		}
	}

	@Override
	public void destroy() throws Exception {
		if(!this.isUseRedis()){
			// 本地缓存
//			if(userCache != null)
//				userCache.clear();
			super.destroy();
		} else {
			// redis 缓存，先清空，2023-08-26
			// 这里不能清空，否则，系统停止时会调用该方法，让redis磁盘中数据清空！ 2023-09-02
//			userCache.clear();
		}
	}

	@Override
	protected int loadDataToCache(Cache cache) {
		return 0;
	}

	@Override
	public T getCacheData(String key) {
		if(this.isUseRedis()){
			return (T)getRedisCache().get(key, getProviderType());
		} else {
			return super.getCacheData(key);
		}
	}

	protected RedisCache getRedisCache(){
		return (RedisCache)this.getCache();
	}

	@Override
	public long getCacheCount(){
		if(this.isUseRedis()){
			return getRedisCache().getPersistentSize();
		} else {
			return super.getCacheCount();
		}
	}

	@Override
	public void removeCacheData(String key) {
		if(this.isUseRedis()){
			getRedisCache().remove(key);
		} else {
			super.removeCacheData(key);
		}
	}

	@Override
	public void updateCacheData(String key, T data) {
		if(this.isUseRedis()){
			getRedisCache().replace(key, data);
		} else {
			super.updateCacheData(key, data);
		}
	}

	@Override
	public void putCacheData(String key, T data){
		if(this.isUseRedis()){
			getRedisCache().put(key, data);
		} else {
			super.putCacheData(key, data);
		}
	}

	/**
	 * 添加支持数据过期的缓存方法。
	 * @param key
	 * @param data
	 * @param expiredSeconds 过期秒数
	 * @date 2022-11-13
	 */
	@Override
	public void putCacheData(String key, T data, long expiredSeconds){
		assert (StringUtils.isNotEmpty(key));
		assert (data != null);
		assert(expiredSeconds > 0);
		if(this.isUseRedis()){
			getRedisCache().put(key, data, expiredSeconds);
		} else {
			super.putCacheData(key, data, expiredSeconds);
		}
	}

	@Override
	public void reload() throws Exception{
		if(this.isUseRedis()){
//			destroy();
//			afterPropertiesSet();
			// 2023-09-04 目前redis重新加载机制和本地缓存一样，都要删除原有缓存数据。
			super.reload();
		} else {
			super.reload();
		}
	}

	/**
	 * 缓存对象不再需要单独根下面的独立的key方式。
	 * @param key
	 * @return
	 * @date 2024-01-05
	 */
	@Deprecated
	public String getRedisRootData(String key){
		return getRedisCache().getRootData(key);
	}

	public boolean isSupportExpiredCache() {
		return supportExpiredCache;
	}

	/**
	 * 设置是否支持缓存失效，因为redis失效是针对key，因此HashMap方式是作为整体失效的！
	 * <pre>
	 *     1) 默认不支持缓存失效，此时使用Map方式存储数据
	 *     2) 当设置运行失效时，必须采用字符串形式的数据结构
	 *     3) 该选项只对hashmap数据结构有效。
	 * </pre>
	 * @param supportExpiredCache
	 * @date 2024-01-05
	 */
	public void setSupportExpiredCache(boolean supportExpiredCache) {
		this.supportExpiredCache = supportExpiredCache;
	}

	private boolean supportExpiredCache = false;
}
