package com.walker.support.redis.cache;

import com.walker.cache.Cache;
import com.walker.infrastructure.utils.StringUtils;

import java.util.List;

/**
 * 支持集合（List）缓存的实现。
 * @param <T>
 * @date 2023-06-14
 */
public abstract class RedisListCacheProvider<T> extends RedisCacheProvider<T>{

    @Override
    protected int loadDataToCache(Cache cache) {
        logger.info("基于集合的Redis缓存，不支持初始化加载缓存数据。");
        return 0;
    }

    @Override
    public T getCacheData(String key) {
        throw new UnsupportedOperationException("不支持该方法，请调用：getCacheList()");
    }

    @Override
    public long getCacheCount(){return 0;}

    @Override
    public void removeCacheData(String key) {
        throw new UnsupportedOperationException("不支持该方法，请调用：removeList()");
    }

    @Override
    public void updateCacheData(String key, T data) {
        throw new UnsupportedOperationException("不支持更新集合方法，请先删除老数据再添加");
    }

    @Override
    public void putCacheData(String key, T data){
        this.putCacheData(key, data, 0);
    }

    @Override
    public void putCacheData(String key, T data, long expiredSeconds){
        this.checkKey(key, data);
        this.getRedisCache().putListAppend(key, data);
    }

    @Override
    public void putCacheList(String key, List<T> data){
        this.putCacheList(key, data, 0);
    }

    @Override
    public void putCacheList(String key, List<T> data, long expiredSeconds){
        if(StringUtils.isEmptyList(data)){
            throw new IllegalArgumentException("list is required!");
        }
        if(StringUtils.isEmpty(key)){
            throw new IllegalArgumentException("key is required!");
        }
        this.getRedisCache().putList(key, (List<Object>)data, expiredSeconds);
    }

    @Override
    public void putCacheListAppend(String key, T data){
        this.checkKey(key, data);
        this.getRedisCache().putListAppend(key, data);
    }

    private void checkKey(String key, T data){
        if(data == null){
            throw new IllegalArgumentException("data is required!");
        }
        if(StringUtils.isEmpty(key)){
            throw new IllegalArgumentException("key is required!");
        }
    }

    @Override
    public List<T> getCacheList(String key){
        if(StringUtils.isEmpty(key)){
            throw new IllegalArgumentException("key is required!");
        }
        return (List<T>)this.getRedisCache().getList(key, 0, -1, getProviderType());
    }

    @Override
    public void removeCacheList(String key, T data){
        this.getRedisCache().removeList(key, data);
    }

    @Override
    public void removeCacheList(String key){
        this.getRedisCache().removeList(key);
    }
}
