package com.walker.support.redis.cache;

import com.walker.cache.Cache;
import com.walker.cache.PageCacheProvider;
import com.walker.db.page.GenericPager;
import com.walker.infrastructure.utils.StringUtils;
import com.walker.support.redis.RedisHelper;

import java.util.Map;

/**
 * 描述：带分页的Redis缓存对象默认实现
 * @author 时克英
 * @date 2017年2月14日 上午10:00:54
 */

public abstract class RedisPageCacheProvider extends RedisCacheProvider<String>
implements PageCacheProvider {

	private RedisHelper redisHelper;

	public void setRedisHelper(RedisHelper redisHelper) {
		this.redisHelper = redisHelper;
	}

	@Override
	protected Cache provideCacheInstance(String name, Map<String, String> param){
		if(this.isUseRedis()){
			RedisCache cache = new RedisPageCache(name, param);
//			cache.setRedisDataSource(redisDataSource);
			cache.setRedisHelper(this.redisHelper);
			return cache;
		} else {
			return super.provideCacheInstance(name, param);
		}
	}

	protected RedisPageCache getRedisCache(){
		return (RedisPageCache)this.getCache();
	}

	/**
	 * 为分页缓存对象，返回一页数据集合
	 * @param pageIndex 页码数，从1开始
	 * @param pageSize 每页数量
	 * @return
	 */
	@Override
	public GenericPager<Object> queryListPage(int pageIndex, int pageSize){
		if(!this.isUseRedis()){
			throw new IllegalArgumentException("'queryListPage'方法必须使用redis缓存实现");
		}
		return this.getRedisCache().queryListPage(pageIndex, pageSize);
	}

	@Override
	public String getCacheData(String key) {
		if(this.isUseRedis()){
			Object data = getRedisCache().get(key);
			return data == null ? StringUtils.EMPTY_STRING : data.toString();
		} else {
			return super.getCacheData(key);
		}
	}

	@Override
	@Deprecated
	public long getCacheCount(){
		return this.getRedisCache().size();
	}

	@Override
	public long size(){
		return this.getRedisCache().size();
	}

	@Override
	public void removeCacheData(String key) {
		if(this.isUseRedis()){
			getRedisCache().removeSort(key);
		} else {
			super.removeCacheData(key);
		}
	}

	@Override
	public void updateCacheData(String key, String data) {
		if(this.isUseRedis()){
			getRedisCache().replace(key, data);
		} else {
			super.updateCacheData(key, data);
		}
	}

	@Override
	public void putCacheData(String key, String data){
		throw new UnsupportedOperationException("分页Redis缓存不支持该方法，请使用'putCacheSort'");
	}

	/**
	 * 为分页缓存对象，添加一个缓存
	 * @param sort 排序值，例如：时间、数值等
	 * @param key 对象唯一id
	 * @param data 存储数据字符串
	 */
	public void putCacheSort(String key, double sort, String data){
		if(this.isUseRedis()){
			getRedisCache().putSort(sort, key, data);
		} else {
			throw new IllegalArgumentException("'putCacheSort'方法必须使用redis缓存实现");
		}
	}

}
