package com.walker.tcp;

import com.walker.infrastructure.utils.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 集中存放系统定义的所有<code>ProtocolResolver</code>对象。</p>
 * 注意：一定要把默认的回车换行的ProtocolResolver配置在最后面，因为有可能其他特殊前缀的报文也会使用回车换行结尾，
 * 因此如果出现这种情况，也能确保用户自定义的ProtocolResolver会优先被调用。
 * @author 时克英
 * @date 2018-11-27
 *
 */
public class ActionCallablePostProcessor implements BeanPostProcessor {

	private final transient Logger logger = LoggerFactory.getLogger(getClass());

	private static Map<String, ActionCallable> reference = new ConcurrentHashMap<>();

//	private static List<ProtocolResolver> cacheList =null;

//	private final InnerComparator comparator = new InnerComparator();

	@Override
	public Object postProcessAfterInitialization(Object bean, String beanName)
			throws BeansException {
		if(ActionCallable.class.isAssignableFrom(bean.getClass())){
			ActionCallable pr = (ActionCallable)bean;
			if(StringUtils.isEmpty(pr.getRequestProtocol())){
				logger.error(pr.getSummary() + "," + pr.getClassName() + ", " + pr.getClass().getName());
				throw new IllegalArgumentException("ActionCallable未实现getRequestProtocol()方法!");
			}
			reference.put(pr.getRequestProtocol(), pr);
			logger.info("找到了一个ActionCallable：" + pr.getClass().getName());
		}

		return bean;
	}

	/**
	 * 根据数据协议编号，查找要处理该业务的action实例。</p>
	 *
	 * @param protocolNum
	 * @return
	 */
	public static final ActionCallable getAction(String protocolNum){
		if(reference.size() == 0){
			throw new IllegalArgumentException(ActionCallable.ERR_NOT_FOUND);
		}

		ActionCallable action = reference.get(protocolNum);
		if(action == null){
			throw new IllegalArgumentException("未找到定义的ActionCallable，protocolNum = " + protocolNum);
		}
		return action;
	}

	public static final List<ActionCallable> getActionList(){
		List<ActionCallable> result = new ArrayList<>();
		for(ActionCallable action : reference.values()){
			result.add(action);
		}
		return result;
	}

	/**
	 * 测试方法。
	 * @param map
	 */
	public static final void setTestActions(Map<String, ActionCallable> map){
		if(ActionCallablePostProcessor.reference.size() > 0){
			System.out.println("++++++++++ " + reference);
			throw new IllegalStateException("reference已经存在数据，不能覆盖。该方法只能在测试时使用!");
		}
		ActionCallablePostProcessor.reference = map;
	}
}
