package com.walker.semantics;

/**
 * 描述：词元对象定义
 * @author 时克英
 * @date 2020年11月10日 上午11:23:10
 */

public class WordMeta {

	private String text;
	private SpeechPart speechPart = null;
	
	private int index = 0;	// 单词所在句子中的索引值，从0开始
	
	// 是否重复数据，最初考虑句子中的词是不重复的，但考虑太单纯，
	// 在分析时间范围过程中，就出现多个月份，但还不能删掉，所以设置此字段表示已经重复的词。
	private boolean duplication = false;
	
	/**
	 * 返回是否已经是重复的词
	 * @return
	 */
	public boolean isDuplication() {
		return duplication;
	}

	public void setDuplication(boolean duplication) {
		this.duplication = duplication;
	}

	public int getIndex() {
		return index;
	}

	public void setIndex(int index) {
		this.index = index;
	}

	public WordMeta(String text, SpeechPart speechPart){
		if(text == null || text.equals("") || speechPart == null){
			throw new IllegalArgumentException("缺少参数：text | speechPart");
		}
		this.text = text;
		this.speechPart = speechPart;
	}
	
	public String getText(){
		return this.text;
	}
	
	public SpeechPart getSpeechPart(){
		return this.speechPart;
	}
	
	/**
	 * 是否系统规定的名词，不包括：道路和建筑
	 * @return
	 * @date 2021-09-01
	 */
	public boolean isMyName(){
		if(this.speechPart == SpeechPart.MY_N){
			return true;
		}
		return false;
	}
	
	/**
	 * 是否系统定义的建筑，如：中原饭店。</p>
	 * 该关键词会提交到系统以及语音系统中
	 * @return
	 * @date 2021-09-01
	 */
	public boolean isMyPlace(){
		if(this.speechPart == SpeechPart.MY_PLACE){
			return true;
		}
		return false;
	}
	
	/**
	 * 是否系统定义的道路，如：文化路。</p>
	 * 该关键词会提交到系统以及语音系统中
	 * @return
	 * @date 2021-09-01
	 */
	public boolean isMyRoad(){
		if(this.speechPart == SpeechPart.MY_ROAD){
			return true;
		}
		return false;
	}
	
	/**
	 * 是否是地点词
	 * @return
	 */
	public boolean isPlace(){
		if(this.speechPart == SpeechPart.MY_PLACE 
				|| this.speechPart == SpeechPart.NS_PLACENAME 
				|| this.speechPart == SpeechPart.S_PLACE){
			return true;
		}
		return false;
	}
	
	/**
	 * 是否已经是自定义指令词。</p>
	 * 指令词不管动词、名词，系统自定义一律使用 SpeechPart.MY_V 类型
	 * @return
	 */
	public boolean isInstruction(){
		if(this.speechPart == SpeechPart.MY_V){
			return true;
		}
		return false;
	}
	
	/**
	 * 是否指令辅助词，如：显示...(分布)
	 * @return
	 */
	public boolean isInstructionAuxiliary(){
		if(this.speechPart == SpeechPart.MY_V_AUX){
			return true;
		}
		return false;
	}
	
	public boolean isResource(){
//		if(this.speechPart == SpeechPart.MY_N){
		// 2021-09-01 修改：把建筑、道路词性，单独区分开了
		if(this.speechPart == SpeechPart.MY_N 
				|| this.isMyPlace() || this.isMyRoad()){
			return true;
		}
		return false;
	}
	
	/**
	 * 可当指令使用的词性
	 * @return
	 */
	public boolean isAvailableInstruction(){
		if(this.speechPart == SpeechPart.V_VERB 
				|| this.speechPart == SpeechPart.VN_VERBNAME 
				|| this.speechPart == SpeechPart.N_NAME
				
				// 注意：在测试中发现【哪里】这个词性是rys，这个很特殊
				// 在定位中使用了
				|| this.speechPart == SpeechPart.RYS){
			return true;
		}
		return false;
	}
	
	/**
	 * 是否动词
	 * @return
	 */
	public boolean isVerbs(){
		if(this.speechPart == SpeechPart.V_VERB 
				|| this.speechPart == SpeechPart.VN_VERBNAME){
			return true;
		}
		return false;
	}
	
	/**
	 * 是否介词</p>
	 * <li>该方法目前在解析时间中判断，如：[从...] 至 使用
	 * @return
	 */
	public boolean isPreposition(){
		if(this.speechPart == SpeechPart.P_PREPOSITION){
			return true;
		}
		return false;
	}
	
	/**
	 * 是否方位词</p>
	 * <li>该方法目前在解析时间中判断，如：...之间
	 * @return
	 */
	public boolean isF_Position(){
		if(this.speechPart == SpeechPart.F_POSITION){
			return true;
		}
		return false;
	}
	
	/**
	 * 是否时间词</p>
	 * <li>该方法目前在解析时间中判断，如：最近
	 * @return
	 */
	public boolean isTime(){
		if(this.speechPart == SpeechPart.T_TIME){
			return true;
		}
		return false;
	}
	
	/**
	 * 是否数值
	 * @return
	 */
	public boolean isNumber(){
		if(this.speechPart == SpeechPart.M_NUMBER){
			return true;
		}
		return false;
	}
	
	/**
	 * 是否数值+单位（数值可能是中文，如：一天、一个月等）
	 * @return
	 */
	public boolean isMQ(){
		if(this.speechPart == SpeechPart.MQ_NUMBER_UNIT){
			return true;
		}
		return false;
	}
	
	/**
	 * 是否单位，如：个，双等
	 * @return
	 */
	public boolean isQ(){
		if(this.speechPart == SpeechPart.Q_QUANTIFIER || this.speechPart == SpeechPart.QV){
			return true;
		}
		return false;
	}
	
	/**
	 * 是否数量，因为情况复杂，发现词性：m和mq是混合存在的，根据句子不同是不确定的，所以这里<br>
	 * 返回的是只要包含数量词相关的情况都返回True，如：
	 * <pre>
	 * m: 五十，3个，三个，五十六个，120个，2万，2万个，两万，两万个
	 * mq：二十二公里，两只，一双，4500人
	 * </pre>
	 * @return
	 */
	public boolean isQuantity(){
		if(this.speechPart == SpeechPart.M_NUMBER || this.speechPart == SpeechPart.MQ_NUMBER_UNIT){
			return true;
		}
		return false;
	}
	
	/**
	 * 数词，如：几、12345等，有些很特殊，如：一个月（一个mq），而两个月、三个月就分开了
	 * @return
	 */
	@Deprecated
	public boolean isM(){
		if(this.speechPart == SpeechPart.M_NUMBER){
			return true;
		}
		return false;
	}
	
	/**
	 * 是否形容词
	 * @return
	 */
	public boolean isAdjective(){
		if(this.speechPart == SpeechPart.A_ADJECTIVE){
			return true;
		}
		return false;
	}
	
	/**
	 * 是否名词(或动名词)
	 * @return
	 */
	public boolean isName(){
		if(this.speechPart == SpeechPart.N_NAME || this.speechPart == SpeechPart.VN_VERBNAME){
			return true;
		}
		return false;
	}
	
	@Override
	public boolean equals(Object obj){
		if(obj == null){
			return false;
		}
		if(obj instanceof WordMeta){
			WordMeta wm = (WordMeta)obj;
			if(wm.text.equals(this.text)){
				return true;
			}
		}
		return false;
	}
	
	@Override
	public int hashCode(){
		return this.text.hashCode();
	}
	
	@Override
	public String toString(){
		return new StringBuilder().append("[text=").append(this.text)
				.append(", speechPart=").append(this.speechPart)
				.append("]").toString();
	}
}
