package com.walker.web;

import com.walker.infrastructure.utils.StringUtils;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;


/**
 * 定义在WEB层异常统一处理的机制。
 * @author 时克英
 * @date 2022-10-12
 */
@RestControllerAdvice
public class GlobalExceptionHandler {

    protected final transient Logger logger = LoggerFactory.getLogger(this.getClass());

    private static final String ERROR_MESSAGE = "系统内部异常";

    /**
     * 定义统一处理业务在WEB中抛出的运行异常。
     * <pre>
     *     1) 在 Controller 层中，通常不会抛出运行异常，只有特定情况下，这是一种处理方式。
     *     2) 在 Controller 层中，开发人员想直接使用 Transaction 数据库事务，此时需要抛出异常中断事务。
     * </pre>
     * @param request
     * @param e
     * @param response
     * @return
     */
    @ExceptionHandler(WebRuntimeException.class)
    public ResponseValue<String> webRuntimeExceptionHandler(HttpServletRequest request, final Exception e, HttpServletResponse response){
        response.setCharacterEncoding(StringUtils.DEFAULT_CHARSET_UTF8);
        response.setContentType(Constants.APPLICATION_JSON);
        response.setStatus(Constants.HTTP_SUCCESS);

        WebRuntimeException webRuntimeException = (WebRuntimeException)e;

        String error = webRuntimeException.getMsg();
        if(StringUtils.isEmpty(error)){
            error = ERROR_MESSAGE;
        }
        logger.error("业务抛出web异常:" + webRuntimeException.getMessage(), e);
        return ResponseValue.error(webRuntimeException.getCode(), error);
    }

    @ExceptionHandler(RuntimeException.class)
    public ResponseValue<String> runtimeExceptionHandler(HttpServletRequest request, final Exception e){
//        response.setCharacterEncoding(StringUtils.DEFAULT_CHARSET_UTF8);
//        response.setContentType(Constants.APPLICATION_JSON);
//        response.setStatus(Constants.HTTP_SUCCESS);
//        WebRuntimeException webRuntimeException = (WebRuntimeException)e;
//        String error = e.getMsg();
//        if(StringUtils.isEmpty(error)){
//            error = ERROR_MESSAGE;
//        }
        logger.error("请求地址'{}',发生运行异常.", request.getRequestURI(), e);
        return ResponseValue.error(e.getMessage());
    }

    @ExceptionHandler(Exception.class)
    public ResponseValue<String> exceptionHandler(HttpServletRequest request, final Exception e){
        logger.error("请求地址'{}',发生系统异常.", request.getRequestURI(), e);
        return ResponseValue.error(e.getMessage());
    }
}
