package com.walker.web;

import com.walker.infrastructure.ApplicationRuntimeException;
import okhttp3.ConnectionPool;
import okhttp3.OkHttpClient;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.OkHttp3ClientHttpRequestFactory;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.web.client.RestTemplate;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import java.nio.charset.StandardCharsets;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.cert.X509Certificate;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * RestTemplate工厂创建对象。
 * @author 时克英
 * @date 2023-08-16
 */
public class RestTemplateFactory {

    public static final RestTemplate createRestTemplate(HttpType httpType, RestTemplateConfig config){
        if(config == null || httpType == null){
            throw new IllegalArgumentException("请提供参数");
        }
        if(httpType == HttpType.OkHttp){
            try {
                return acquireHttpOkRestTemplate(config.getMaxIdleConnections(), config.getKeepAliveDurationSeconds()
                        , config.getConnectTimeoutSeconds(), config.getReadTimeoutSeconds(), config.getWriteTimeoutSeconds());
            } catch (Exception e) {
                throw new ApplicationRuntimeException("创建：RestTemplate失败, " + e.getMessage(), e);
            }
        } else {
            throw new UnsupportedOperationException("未实现其他方式http连接");
        }
    }

    private static RestTemplate acquireHttpOkRestTemplate(int maxIdleConnections
            , long keepAliveDuration, long connectTimeout, long readTimeout, long writeTimeout) throws Exception{
        ConnectionPool connectionPool = new ConnectionPool(maxIdleConnections, keepAliveDuration, TimeUnit.SECONDS);
        OkHttpClient client = new OkHttpClient().newBuilder()
                .connectionPool(connectionPool)
                .connectTimeout(connectTimeout, TimeUnit.SECONDS)
                .readTimeout(readTimeout, TimeUnit.SECONDS)
                .writeTimeout(writeTimeout, TimeUnit.SECONDS)
//                .hostnameVerifier((hostname, session) -> true)
                .sslSocketFactory(getIgnoreInitSslContext().getSocketFactory(), IGNORE_SSL_TRUST_MANAGER_X509)
                .hostnameVerifier(getIgnoreSslHostnameVerifier())   // ssl相关，忽略否则https调用报证书错误
                // 设置代理
//              .proxy(new Proxy(Proxy.Type.HTTP, new InetSocketAddress("127.0.0.1", 8888)))
                // 拦截器
//                .addInterceptor()
                .build();

        ClientHttpRequestFactory factory = new OkHttp3ClientHttpRequestFactory(client);
        RestTemplate restTemplate = new RestTemplate(factory);
        setupRestTemplate(restTemplate);
        return restTemplate;
    }

    private static void setupRestTemplate(RestTemplate restTemplate){
        List<HttpMessageConverter<?>> messageConverters = restTemplate.getMessageConverters();
        Iterator<HttpMessageConverter<?>> iterator = messageConverters.iterator();
        while(iterator.hasNext()){
            HttpMessageConverter<?> converter=iterator.next();
            //原有的String是ISO-8859-1编码 去掉
            if(converter instanceof StringHttpMessageConverter){
                iterator.remove();
            }
        }
        messageConverters.add(new StringHttpMessageConverter(StandardCharsets.UTF_8));
    }

    /**
     * Get initialized SSLContext instance which ignored SSL certification
     *
     * @return
     * @throws NoSuchAlgorithmException
     * @throws KeyManagementException
     */
    public static SSLContext getIgnoreInitSslContext() throws NoSuchAlgorithmException, KeyManagementException {
        SSLContext sslContext = SSLContext.getInstance("SSL");
        sslContext.init(null, new TrustManager[] { IGNORE_SSL_TRUST_MANAGER_X509 }, new SecureRandom());
        return sslContext;
    }

    /**
     * X509TrustManager instance which ignored SSL certification
     */
    public static final X509TrustManager IGNORE_SSL_TRUST_MANAGER_X509 = new X509TrustManager() {
        @Override
        public void checkClientTrusted(X509Certificate[] chain, String authType) {
        }

        @Override
        public void checkServerTrusted(X509Certificate[] chain, String authType) {
        }

        @Override
        public X509Certificate[] getAcceptedIssuers() {
            return new X509Certificate[] {};
        }
    };

    public static HostnameVerifier getIgnoreSslHostnameVerifier() {
        return new HostnameVerifier() {
            @Override
            public boolean verify(String arg0, SSLSession arg1) {
                return true;
            }
        };
    }

    public enum HttpType{
        OkHttp,
        HttpClient;
    }

//    public class Config{
//
//        public int getMaxIdleConnections() {
//            return maxIdleConnections;
//        }
//
//        public void setMaxIdleConnections(int maxIdleConnections) {
//            this.maxIdleConnections = maxIdleConnections;
//        }
//
//        public long getKeepAliveDurationSeconds() {
//            return keepAliveDurationSeconds;
//        }
//
//        public void setKeepAliveDurationSeconds(long keepAliveDurationSeconds) {
//            this.keepAliveDurationSeconds = keepAliveDurationSeconds;
//        }
//
//        public long getConnectTimeoutSeconds() {
//            return connectTimeoutSeconds;
//        }
//
//        public void setConnectTimeoutSeconds(long connectTimeoutSeconds) {
//            this.connectTimeoutSeconds = connectTimeoutSeconds;
//        }
//
//        public long getReadTimeoutSeconds() {
//            return readTimeoutSeconds;
//        }
//
//        public void setReadTimeoutSeconds(long readTimeoutSeconds) {
//            this.readTimeoutSeconds = readTimeoutSeconds;
//        }
//
//        public long getWriteTimeoutSeconds() {
//            return writeTimeoutSeconds;
//        }
//
//        public void setWriteTimeoutSeconds(long writeTimeoutSeconds) {
//            this.writeTimeoutSeconds = writeTimeoutSeconds;
//        }
//
//        // 连接池中整体的空闲连接的最大数量
//        private int maxIdleConnections = 100;
//        // 连接空闲时间最多为 300 秒
//        private long keepAliveDurationSeconds = 300;
//        // 连接超时（秒）
//        private long connectTimeoutSeconds = 3;
//        // 读超时（秒）
//        private long readTimeoutSeconds = 3;
//        // 写超时（秒）
//        private long writeTimeoutSeconds = 3;
//    }
}
