package com.walker.web;

import java.io.Serializable;
import java.util.List;

/**
 * 用户认证后的存储信息，这里定义规范，由业务实现。
 * <pre>
 *     1）单应用集成 walker-web-security 时需要实现。
 *     2）网关微服务应用 过滤器拦截并返回该对象实现。
 * </pre>
 */
public interface UserPrincipal<T> extends Serializable {

    /**
     * 返回用户唯一编号，业务对应用户数据库中的id
     * @return
     */
    String getId();

    /**
     * 返回用户名称，姓名或昵称
     * @return
     */
    String getUserName();

    /**
     * 返回用户加密后的密码(MD5加密值)
     * @return
     */
    String getPassword();

    /**
     * 用户当前是否可用
     * @return
     */
    boolean isEnabled();

    /**
     * 判断token是否失效
     * @param token 系统生成的唯一token
     * @return
     */
    boolean isTokenExpired(String token);

    /**
     * 用户是否被锁定
     * @return
     */
    boolean isAccountLocked();

    /**
     * 验证MD5密码是否正确
     * @param encryption
     * @return
     */
    boolean validateMd5Password(String encryption);

    /**
     * 返回用户实际对象，通常是业务提供的用户信息。
     * @return
     */
    T getUserInfo();

    /**
     * 上次登录时间，格式: 20221029113601
     * @return
     */
    long getLastLoginTime();

    /**
     * 上次登录类型
     * @return
     */
    LoginType getLastLoginType();

    /**
     * 返回用户所属角色ID集合。为若依前端适配添加。
     * @return
     * @date 2022-11-11
     */
    List<String> getRoleIdList();

    public void setRoleIdList(List<String> roleIdList);

    /**
     * 是否超级管理员
     * @return
     * @date 2022-12-21
     */
    boolean isSupervisor();

    /**
     * 返回用户对应某个功能的数据权限标识。
     * @param menuId
     * @return
     * @date 2022-12-21
     */
    String getDataScope(String menuId);
}
