/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-algorithm project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.algorithm;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Spliterator;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Stream;
import terraml.algorithm.iterator.BiOrderedTraversal;
import terraml.algorithm.node.BiNode;
import terraml.algorithm.node.BiSearchNode;
import static terraml.commons.Objects.isNull;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public class BiSearch<Q> implements BiTreeSet<Q>, Serializable {

    // bir kürsüm yok fakat ulaştım herkese özetlerimle geldim.
    private static final long serialVersionUID = 9273273515765231L;

    private final Comparator<Q> comparator;
    private BiNode<Q> root;

    /**
     * @param comp
     * @param item
     */
    public BiSearch(Comparator<Q> comp, Q item) {
        this.comparator = comp;
        this.root = new BiSearchNode<>(comparator, item);
    }

    /**
     * @param comparator
     * @param root
     */
    public BiSearch(Comparator<Q> comparator, BiNode<Q> root) {
        this.comparator = comparator;
        this.root = root;
    }

    /**
     * @param src source
     * @return true if given node has duplicate nodes. Otherwise it returns false.
     */
    protected boolean hasDuplicates(BiNode<Q> src) {
        return src != null && src.getDuplicates() != null;
    }

    /**
     * @param data comparable object
     * @param src  source
     * @return true if comparison result equals to 0. Otherwise it returns false. Also
     *         if any given arguments is null then it returns false.
     */
    protected boolean eq(Q data, BiNode<Q> src) {
        return data != null && src != null && comparator.compare(data, src.getData()) == 0;
    }

    /**
     * @param data comparable object
     * @param src  source
     * @return true if comparison result smaller than 0. Otherwise it returns false. Also
     *         if any given arguments is null then it returns false.
     */
    protected boolean st(Q data, BiNode<Q> src) {
        return data != null && src != null && comparator.compare(data, src.getData()) < 0;
    }

    /**
     * @param data comparable object
     * @param src  source
     * @return true if comparison result greater than 0. Otherwise it returns false. Also
     *         if any given arguments is null then it returns false.
     */
    protected boolean gt(Q data, BiNode<Q> src) {
        return data != null && src != null && comparator.compare(data, src.getData()) > 0;
    }

    /**
     * @param data comparable object
     * @param src  source
     * @return true if comparison result greater than 0. Otherwise it returns false. Also
     *         if any given arguments is null then it returns false.
     */
    protected boolean btw(Q from, Q to, BiNode<Q> src) {
        if (isNull(from) || isNull(to) || isNull(src)) {
            return false;
        }

        return st(from, src) && gt(to, src);
    }

    /**
     * @param src source
     * @return true if given nodes left child is empty. Otherwise it returns false. Also
     *         if given node is null then it returns false.
     */
    protected boolean le(BiNode<Q> src) {
        return src != null && src.getLeft() == null;
    }

    /**
     * @param src source
     * @return true if given nodes right child is empty. Otherwise it returns false. Also
     *         if given node is null then it returns false.
     */
    protected boolean re(BiNode<Q> src) {
        return src != null && src.getRight() == null;
    }

    /**
     * @param src source
     * @return true if given nodes left and right child is empty(leaf). Otherwise it returns false. Also
     *         if given node is null then it returns false.
     */
    protected boolean lf(BiNode<Q> src) {
        return src != null && src.getLeft() == null && src.getRight() == null;
    }

    /**
     * @param src source
     * @return count of elements on given node. It counts if any duplicates exists.
     */
    protected int size(BiNode<Q> src) {
        if (src == null) {
            return 0;
        }

        int count = 1;
        if (hasDuplicates(src)) {
            count += src.getDuplicates().size();
        }

        return count + size(src.getLeft()) + size(src.getRight());
    }

    /**
     * @param src source
     * @return count of elements on given node. It doesn't count if any duplicates exists.
     */
    protected int familySize(BiNode<Q> src) {
        if (src == null) {
            return 0;
        }

        return 1 + familySize(src.getLeft()) + familySize(src.getRight());
    }

    /**
     * @param src source
     * @return height of given node.
     */
    protected int height(BiNode<Q> src) {
        if (src == null) {
            return 0;
        }

        return 1 + Math.max(height(src.getLeft()), height(src.getRight()));
    }

    /**
     * @param src source
     * @return minimum comparable object.
     */
    protected Q minimum(BiNode<Q> src) {
        if (le(src)) {
            return src.getData();
        }

        return minimum(src.getLeft());
    }

    /**
     * @param src source
     * @return maximum comparable object.
     */
    protected Q maximum(BiNode<Q> src) {
        if (re(src)) {
            return src.getData();
        }

        return maximum(src.getRight());
    }

    /**
     * @param src source
     * @return node that hold minimum comparable objects.
     */
    protected BiNode<Q> minimumFamily(BiNode<Q> src) {
        if (le(src)) {
            return src;
        }

        return minimumFamily(src.getLeft());
    }

    /**
     * @param src source
     * @return node that hold maximum comparable objects.
     */
    protected BiNode<Q> maximumFamily(BiNode<Q> src) {
        if (re(src)) {
            return src;
        }

        return maximumFamily(src.getRight());
    }

    /**
     * @param src  source
     * @param item comparable object
     * @return a new node that containing all given elements on given node plus
     *         a new one that created with given item.
     */
    protected BiNode<Q> add(BiNode<Q> src, Q item) {
        if (src == null) {
            return new BiSearchNode<>(comparator, item);
        } else if (eq(item, src)) {
            src.addDuplicate(item);
            return src;
        } else if (st(item, src)) {
            src.setLeft(add(src.getLeft(), item));
        } else if (gt(item, src)) {
            src.setRight(add(src.getRight(), item));
        }

        return src;
    }

    @Override
    public boolean add(Q item) {
        final BiNode<Q> subRoot = add(getRoot(), item);

        if (subRoot != null) {

            setRoot(subRoot);
            return true;

        }

        return false;
    }

    @Override
    public boolean addAll(Collection<? extends Q> collection) {
        boolean isOk = true;

        for ( Q each : collection ) {

            if (!add(each)) {
                isOk = false;
                break;
            }

        }

        return isOk;
    }

    /**
     * @param src  source
     * @param item comparable object
     * @return a new node that containing all elements from given node
     *         except given items family.
     */
    protected BiNode<Q> removeFamily(BiNode<Q> src, Q item) {
        if (eq(item, src)) {
            if (lf(src)) {
                return null;
            } else if (le(src)) {
                return src.getRight();
            } else if (re(src)) {
                return src.getLeft();
            }

            BiNode<Q> edge = src.getRight();
            BiNode<Q> current = src.getRight();

            while ( !le(edge) ) {
                edge = edge.getLeft();
            }

            edge.setLeft(src.getLeft());
            return current;
        } else if (st(item, src)) {
            src.setLeft(removeFamily(src.getLeft(), item));
        } else if (gt(item, src)) {
            src.setRight(removeFamily(src.getRight(), item));
        }

        return src;
    }

    @Override
    public void removeFamily(Q item) {
        final BiNode<Q> tmpFamily = removeFamily(getRoot(), item);

        if (tmpFamily != null) {
            setRoot(tmpFamily);
        }
    }

    @Override
    public void removeAllFamily(Collection<? extends Q> collection) {
        collection.forEach(this::removeFamily);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean remove(Object item) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean removeAll(Collection<?> collection) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * @param src  source
     * @param item comparable object
     * @return a node that contains given comparable objects.
     */
    protected BiNode<Q> getFamily(BiNode<Q> src, Q item) {
        if (src == null) {
            return null;
        }

        if (st(item, src)) {
            src = getFamily(src.getLeft(), item);
        } else if (gt(item, src)) {
            src = getFamily(src.getRight(), item);
        }

        return eq(item, src) ? src : null;
    }

    @Override
    public boolean containsFamily(Q item) {
        return getFamily(getRoot(), item) != null;
    }

    @Override
    public boolean containsAllFamily(Collection<Q> collection) {

        for ( Q each : collection ) {

            if (!containsFamily(each)) {
                return false;
            }

        }

        return true;
    }

    /**
     * @param src  source
     * @param item comparable object
     * @param <Q>  generi type
     * @return a node that equals to given element.
     */
    protected BiNode<Q> get(BiNode<Q> src, Q item) {
        if (src == null) {
            return null;
        }

        if (st(item, src)) {
            // branch left
            src = get(src.getLeft(), item);
        } else if (gt(item, src)) {
            // branch right
            src = get(src.getRight(), item);
        }

        if (eq(item, src)) {

            // if item eq to current node data, then current node gonna be returned.
            if (item.equals(src.getData())) {
                return src;
            }

            // if item neq and current node has dupplicates
            if (hasDuplicates(src)) {

                // check every dupplicate for equality
                for ( Q each : src.getDuplicates() ) {

                    // if item eq one of the dupplicates then current node gonna be returned
                    if (item.equals(each)) {
                        return src;
                    }

                }

            }

        }

        return null;
    }

    @Override
    public boolean contains(Object item) {
        return get(getRoot(), (Q) item) != null;
    }

    @Override
    public boolean containsAll(Collection<?> collection) {
        for ( Object each : collection ) {

            if (!contains(each)) {
                return false;
            }

        }

        return true;
    }

    @Override
    public int size() {
        return size(getRoot());
    }

    @Override
    public Iterator<Q> lastFamily() {
        // find last
        BiNode<Q> bi = minimumFamily(getRoot());

        List<Q> list = new ArrayList<>();

        // add primary data
        list.add(bi.getData());

        // if there are duplicates
        if (hasDuplicates(bi)) {

            // add each one of them to the list.
            for ( Q each : bi.getDuplicates() ) {
                list.add(each);
            }

        }

        return list.iterator();
    }

    @Override
    public Iterator<Q> firstFamily() {
        // find first
        BiNode<Q> bi = maximumFamily(getRoot());

        List<Q> list = new ArrayList<>();

        // add primary data
        list.add(bi.getData());

        // if there are duplicates
        if (hasDuplicates(bi)) {

            // add each one of them to the list.
            for ( Q each : bi.getDuplicates() ) {
                list.add(each);
            }

        }

        return list.iterator();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Q last() {
        return maximum(getRoot());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Q first() {
        return minimum(getRoot());
    }

    /**
     * @param src
     * @param list
     * @return
     */
    protected List<Q> collect(BiNode<Q> src, List<Q> list) {
        if (src == null) {
            return list;
        }

        // add head node data
        list.add(src.getData());

        // if node has duplicates
        if (hasDuplicates(src)) {

            // add all
            for ( Q each : src.getDuplicates() ) {
                list.add(each);
            }

        }

        // branch left
        list = collect(src.getLeft(), list);

        // branch right
        list = collect(src.getRight(), list);

        return list;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Iterator<Q> iterator() {
        return new BiOrderedTraversal<>(getRoot());
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Stream<Q> stream() {
        return collect(root, new ArrayList<>()).stream();
    }

    /**
     * <p>
     * Basic getter method.
     * </p>
     *
     * @return root node.
     */
    public BiNode<Q> getRoot() {
        return this.root;
    }

    /**
     * <p>
     * Basic setter method.
     * </p>
     *
     * @param obj root object
     */
    public void setRoot(BiNode<Q> obj) {
        this.root = obj;
    }

    @Override
    public Spliterator<Q> spliterator() {
        return collect(root, new ArrayList<>()).spliterator();
    }

    /**
     * @param src
     * @param item
     * @return
     */
    protected BiNode<Q> tailSet(BiNode<Q> src, Q item) {
        if (src == null) {
            return null;
        }

        while ( (eq(item, src) || gt(item, src)) && src != null ) {
            src = tailSet(src.getRight(), item);
        }

        if (st(item, src)) {
            src.setLeft(tailSet(src.getLeft(), item));
        }

        return src;
    }

    @Override
    public BiTreeSet<Q> tailSet(Q fromElement) {
        BiNode<Q> bi = getRoot();
        bi = tailSet(bi, fromElement);

        return new BiSearch<>(comparator, bi);
    }

    /**
     * @param src
     * @param item
     * @return
     */
    protected BiNode<Q> headSet(BiNode<Q> src, Q item) {
        if (src == null) {
            return null;
        }

        while ( (eq(item, src) || st(item, src)) && src != null ) {
            src = headSet(src.getLeft(), item);
        }

        if (gt(item, src)) {
            src.setRight(headSet(src.getRight(), item));
        }

        return src;
    }

    @Override
    public BiTreeSet<Q> headSet(Q toElement) {
        BiNode<Q> bi = getRoot();
        bi = headSet(bi, toElement);

        return new BiSearch<>(comparator, bi);
    }

    /**
     * @param src
     * @param from
     * @param to
     * @return
     */
    protected BiNode<Q> subSet(BiNode<Q> src, Q from, Q to) {
        if (src == null) {
            return null;
        }

        if (st(to, src)) {
            return subSet(src.getLeft(), from, to);
        } else if (gt(from, src)) {
            return subSet(src.getRight(), from, to);
        }

        src.setLeft(subSet(src.getLeft(), from, to));
        src.setRight(subSet(src.getRight(), from, to));

        return src;
    }

    @Override
    public BiTreeSet<Q> subSet(Q fromElement, Q toElement) {
        BiNode<Q> bi = getRoot();
        bi = subSet(bi, fromElement, toElement);

        return new BiSearch<>(comparator, bi);
    }

    @Override
    public Comparator<? super Q> comparator() {
        return this.comparator;
    }

    @Override
    public void clear() {
        setRoot(null);
    }

    @Override
    public boolean retainAll(Collection<?> collection) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public <T> T[] toArray(T[] a) {
        return collect(root, new ArrayList<>()).toArray(a);
    }

    @Override
    public Object[] toArray() {
        return collect(root, new ArrayList<>()).toArray();
    }

    @Override
    public boolean isEmpty() {
        return size() == 0;
    }

    @Override
    public Stream<Q> parallelStream() {
        return collect(root, new ArrayList<>()).parallelStream();
    }

    @Override
    public boolean removeIf(Predicate<? super Q> filter) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public void forEach(Consumer<? super Q> action) {
        stream().forEach(action);
    }
}
