/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-algorithm project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.algorithm;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.NoSuchElementException;
import java.util.TreeSet;
import terraml.commons.Doubles;
import static terraml.commons.Objects.isNull;
import static terraml.commons.Objects.nonNull;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version PUBLIC-1.0
 *
 */
public class Dijkstra<Q> {

    // çok sıkıldılar, hiç konuşmadım.
    private final HashMap<Q, DijkstraVertex<Q>> graph;

    private Dijkstra(HashMap<Q, DijkstraVertex<Q>> graph) {
        this.graph = graph;
    }

    /**
     *
     * @param <Q>
     * @param edgeCollection
     * @return
     */
    public static <Q> Dijkstra<Q> create(Collection<DefaultWeightedEdge<Q>> edgeCollection) {
        return new Dijkstra<>(init(edgeCollection));
    }

    /**
     *
     * @param edgeCollection
     */
    private static <Q> HashMap<Q, DijkstraVertex<Q>> init(Collection<DefaultWeightedEdge<Q>> edgeCollection) {
        HashMap<Q, DijkstraVertex<Q>> newGraph = new HashMap<>(edgeCollection.size());

        for ( DefaultWeightedEdge<Q> currentEdge : edgeCollection ) {
            Q object = currentEdge.getSource();
            boolean hasSource = nonNull(newGraph.get(object));

            if (!hasSource) {
                newGraph.put(object, new DijkstraVertex<>(object));
            }

            object = currentEdge.getTarget();
            boolean hasTarget = nonNull(newGraph.get(object));

            if (!hasTarget) {
                newGraph.put(object, new DijkstraVertex<>(object));
            }
        }

        for ( DefaultWeightedEdge<Q> currentEdge : edgeCollection ) {
            Q src = currentEdge.getSource();
            Q tar = currentEdge.getTarget();
            DijkstraVertex<Q> object = newGraph.get(tar);

            newGraph.get(src).getConnections().put(object, currentEdge.getWeight());
        }

        return newGraph;
    }

    /**
     *
     * @return
     */
    public List<Map.Entry<Q, Double>> path() {
        final List<Map.Entry<Q, Double>> list = new ArrayList<>();

        for ( DijkstraVertex<Q> vertex : graph.values() ) {
            list.add(vertex.toEntry());
        }

        return list;
    }

    /**
     *
     * @param pointer
     * @throws NoSuchElementException
     */
    public void dijkstra(final Q pointer) throws NoSuchElementException {
        if (isNull(graph.get(pointer))) {
            throw new NoSuchElementException("Given start pointer is not found in the graph");
        }

        final DijkstraVertex<Q> source = graph.get(pointer);
        NavigableSet<DijkstraVertex<Q>> navigableSet = new TreeSet<>();

        for ( DijkstraVertex<Q> current : graph.values() ) {
            current.setPrevious(current == source ? source : null);
            current.setWeight(current == source ? 0 : Double.MAX_VALUE);

            navigableSet.add(current);
        }

        dijkstra(navigableSet);
    }

    /**
     *
     * @param navigable
     */
    protected void dijkstra(final NavigableSet<DijkstraVertex<Q>> navigable) {
        DijkstraVertex<Q> front, rear;
        while ( !navigable.isEmpty() ) {

            front = navigable.pollFirst();
            if (Doubles.isEqual(front.getWeight(), Double.MAX_VALUE)) {
                break;
            }

            for ( Map.Entry<DijkstraVertex<Q>, Double> entry : front.getConnections().entrySet() ) {
                rear = entry.getKey();

                final double alternative = front.getWeight() + entry.getValue();
                if (alternative < rear.getWeight()) {
                    navigable.remove(rear);

                    rear.setWeight(alternative);
                    rear.setPrevious(front);

                    navigable.add(rear);
                }
            }
        }
    }

    /**
     *
     * @return
     */
    public HashMap<Q, DijkstraVertex<Q>> getGraph() {
        return graph;
    }
}
