/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-algorithm project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.algorithm;

import static java.lang.Math.abs;
import static java.lang.Math.atan;
import static java.lang.Math.atan2;
import static java.lang.Math.cos;
import static java.lang.Math.sin;
import static java.lang.Math.sqrt;
import static java.lang.Math.tan;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@FunctionalInterface
public interface DistanceCalc {

    /**
     * @param p0
     * @param p1
     * @return
     */
    public double calc(GeoPoint p0, GeoPoint p1);

    /**
     * @param x0
     * @param y0
     * @param x1
     * @param y1
     * @return
     */
    default double calc(double x0, double y0, double x1, double y1) {
        // kuponla adam olacak adam hoşgeldin.
        return calc(new GeoPoint(x0, y0), new GeoPoint(x1, y1));
    }

    /**
     * @param p0
     * @param p1
     * @return
     */
    default double vincenty(GeoPoint p0, GeoPoint p1) {
        return vincenty(p0.getX(), p0.getY(), p1.getX(), p1.getY());
    }

    /**
     * @param x0
     * @param y0
     * @param x1
     * @param y1
     * @return
     */
    default double vincenty(double x0, double y0, double x1, double y1) {
        final double lat0 = Math.toRadians(x0);
        final double lon0 = Math.toRadians(y0);
        final double lat1 = Math.toRadians(x1);
        final double lon1 = Math.toRadians(y1);

        return vincentyFromRadian(lat0, lon0, lat1, lon1);
    }

    /**
     * @param lat0
     * @param lon0
     * @param lat1
     * @param lon1
     * @return
     */
    default double vincentyFromRadian(double lat0, double lon0, double lat1, double lon1) {
        double a = 6378137, b = 6356752.314245, f = 1 / 298.257223563;
        double L = lon1 - lon0;
        double U1 = atan((1 - f) * tan(lat0));
        double U2 = atan((1 - f) * tan(lat1));
        double sinU1 = sin(U1), cosU1 = cos(U1);
        double sinU2 = sin(U2), cosU2 = cos(U2);

        double sinLambda, cosLambda, sinSigma, cosSigma, sigma, sinAlpha, cosSqAlpha, cos2SigmaM;
        double lambda = L, lambdaP, iterLimit = 100;
        do {
            sinLambda = sin(lambda);
            cosLambda = cos(lambda);
            sinSigma = sqrt((cosU2 * sinLambda) * (cosU2 * sinLambda)
                    + (cosU1 * sinU2 - sinU1 * cosU2 * cosLambda) * (cosU1 * sinU2 - sinU1 * cosU2 * cosLambda));
            if (sinSigma == 0) {
                return 0;
            }
            cosSigma = sinU1 * sinU2 + cosU1 * cosU2 * cosLambda;
            sigma = atan2(sinSigma, cosSigma);
            sinAlpha = cosU1 * cosU2 * sinLambda / sinSigma;
            cosSqAlpha = 1 - sinAlpha * sinAlpha;
            cos2SigmaM = cosSigma - 2 * sinU1 * sinU2 / cosSqAlpha;
            if (Double.isNaN(cos2SigmaM)) {
                cos2SigmaM = 0;
            }
            double C = f / 16 * cosSqAlpha * (4 + f * (4 - 3 * cosSqAlpha));
            lambdaP = lambda;
            lambda = L + (1 - C) * f * sinAlpha
                    * (sigma + C * sinSigma * (cos2SigmaM + C * cosSigma * (-1 + 2 * cos2SigmaM * cos2SigmaM)));
        } while ( abs(lambda - lambdaP) > 1e-12 && --iterLimit > 0 );

        if (iterLimit == 0) {
            return Double.NaN;
        }
        double uSq = cosSqAlpha * (a * a - b * b) / (b * b);
        double A = 1 + uSq / 16384 * (4096 + uSq * (-768 + uSq * (320 - 175 * uSq)));
        double B = uSq / 1024 * (256 + uSq * (-128 + uSq * (74 - 47 * uSq)));
        double deltaSigma = B
                * sinSigma
                * (cos2SigmaM + B
                   / 4
                   * (cosSigma * (-1 + 2 * cos2SigmaM * cos2SigmaM) - B / 6 * cos2SigmaM
                      * (-3 + 4 * sinSigma * sinSigma) * (-3 + 4 * cos2SigmaM * cos2SigmaM)));

        return b * A * (sigma - deltaSigma);
    }
}
