/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-algorithm project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.algorithm;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version PUBLIC-1.0
 */
public class FloydWarshall<Q> {

    // ilham veririm herkese, hoca kısa kes.
    public static class State<W> {

        public final HashMap<W, List<DefaultWeightedEdge<W>>> graph;
        public final HashMap<W, Integer> index;
        public final int uniqueIdCount;

        public State(HashMap<W, List<DefaultWeightedEdge<W>>> graph, HashMap<W, Integer> index, int pointer) {
            this.graph = graph;
            this.index = index;
            this.uniqueIdCount = pointer;
        }

    }

    private final State<Q> state;
    private final double[][] costMatrix;

    /**
     *
     * @param state
     */
    public FloydWarshall(State<Q> state) {
        this.state = state;
        this.costMatrix = createCostMatrix(this.state);
    }

    /**
     *
     * @param <W>
     * @param collection
     * @return
     */
    public static <W> FloydWarshall<W> create(Collection<DefaultWeightedEdge<W>> collection) {
        return new FloydWarshall<>(init(collection));
    }

    /**
     *
     */
    public void execute() {
        final int len = costMatrix.length;

        for ( int k = 0; k < len; k++ ) {
            for ( int i = 0; i < len; i++ ) {
                for ( int j = 0; j < len; j++ ) {
                    if (costMatrix[i][k] + costMatrix[k][j] < costMatrix[i][j]) {
                        costMatrix[i][j] = costMatrix[i][k] + costMatrix[k][j];
                    }
                }
            }
        }
    }

    /**
     *
     * @return
     */
    public List<DefaultWeightedEdge<Q>> toList() {
        List<DefaultWeightedEdge<Q>> list = new ArrayList<>();

        for ( List<DefaultWeightedEdge<Q>> current : state.graph.values() ) {

            if (current != null) {
                boolean isEmpty = current.isEmpty();

                if (!isEmpty) {
                    list.addAll(current);
                }
            }

        }

        return list;
    }

    /**
     *
     * @param from
     * @param to
     * @return
     */
    public double get(Q from, Q to) {
        int row = state.index.get(from);
        int column = state.index.get(to);

        return get(row, column);
    }

    /**
     *
     * @param row
     * @param column
     * @return
     */
    public double get(int row, int column) {
        return costMatrix[row][column];
    }

    /**
     *
     * @param givenState
     * @return
     */
    private double[][] createCostMatrix(State<Q> givenState) {
        final int len = givenState.uniqueIdCount;
        final double[][] matrix = new double[len][len];

        int indexerSize = givenState.index.size();

        for ( int i = 0; i < indexerSize; i++ ) {
            for ( int j = 0; j < indexerSize; j++ ) {
                matrix[i][j] = Double.NaN;
            }
        }

        for ( int i = 0; i < indexerSize; i++ ) {
            Q current = findData(i);
            List<DefaultWeightedEdge<Q>> tmp = givenState.graph.get(current);

            for ( DefaultWeightedEdge<Q> edge : tmp ) {
                int row = givenState.index.get(edge.getSource());
                int column = givenState.index.get(edge.getTarget());

                matrix[row][column] = edge.getWeight();
            }
        }

        return matrix;
    }

    /**
     *
     * @param index
     * @return
     */
    public Q findData(Integer index) {
        for ( Map.Entry<Q, Integer> entry : state.index.entrySet() ) {
            if (Objects.equals(index, entry.getValue())) {
                return entry.getKey();
            }
        }

        return null;
    }

    /**
     *
     * @param <W>
     * @param collection
     * @return
     */
    private static <W> State<W> init(Collection<DefaultWeightedEdge<W>> collection) {
        HashMap<W, List<DefaultWeightedEdge<W>>> newGraph = new HashMap<>();
        HashMap<W, Integer> indexer = new HashMap<>();
        HashSet<W> pointers = new HashSet<>();

        int index = 0;
        for ( DefaultWeightedEdge<W> currentEdge : collection ) {
            if (!pointers.contains(currentEdge.getSource())) {
                pointers.add(currentEdge.getSource());
            }

            if (!pointers.contains(currentEdge.getTarget())) {
                pointers.add(currentEdge.getTarget());
            }

            W source = currentEdge.getSource();
            W target = currentEdge.getTarget();

            if (newGraph.get(source) == null) {
                List<DefaultWeightedEdge<W>> tmp = new ArrayList<>();
                tmp.add(currentEdge);

                newGraph.put(source, tmp);
                indexer.put(source, index);
                index += 1;
            } else {
                if (!newGraph.get(source).contains(currentEdge)) {
                    newGraph.get(source).add(currentEdge);
                }
            }

            if (newGraph.get(target) == null) {
                newGraph.put(target, new ArrayList<>());

                indexer.put(target, index);
                index += 1;
            }
        }

        return new State<>(newGraph, indexer, pointers.size());
    }

    /**
     *
     * @return
     */
    public double[][] getCostMatrix() {
        return costMatrix;
    }

    /**
     *
     * @return
     */
    public State<Q> getState() {
        return state;
    }

}
