/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-algorithm project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.algorithm;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import terraml.algorithm.iterator.KDEnumerator;
import terraml.algorithm.node.KDNode2;
import static terraml.commons.Objects.nonNull;

/**
 *
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public class KDTree2 implements Collection<GeoPoint>, Iterable<GeoPoint> {

    // kıstırıyorlar seni sokakta, 
    // öldün her köşe başında.
    public static final int K = 2;

    @SuppressWarnings("Convert2Lambda")
    private DistanceCalc distanceCalc = new DistanceCalc() {
        @Override
        public double calc(GeoPoint p0, GeoPoint p1) {
            return vincenty(p0, p1);
        }
    };

    @SuppressWarnings("Convert2Lambda")
    private Comparator<GeoPoint> xComparator = new LatlonComparator() {
        @Override
        public int compare(GeoPoint p0, GeoPoint p1) {
            return latitudeDirectionCompare(p0, p1);
        }
    };

    @SuppressWarnings("Convert2Lambda")
    private Comparator<GeoPoint> yComparator = new LatlonComparator() {
        @Override
        public int compare(GeoPoint p0, GeoPoint p1) {
            return longitudeDirectionCompare(p0, p1);
        }
    };

    private KDNode2 root = null;

    /**
     * @param calc
     * @param xComp
     * @param yComp
     */
    private KDTree2(DistanceCalc calc, Comparator<GeoPoint> xComp, Comparator<GeoPoint> yComp) {
        if (nonNull(calc)) {
            this.distanceCalc = calc;
        }

        if (nonNull(xComp)) {
            this.xComparator = xComp;
        }

        if (nonNull(yComp)) {
            this.yComparator = yComp;
        }
    }

    /**
     * @param builder
     */
    public KDTree2(KDTree2Builder builder) {
        this(builder.getDistanceCalc(), builder.getxComparator(), builder.getyComparator());
    }

    public KDTree2() {
        this(null, null, null);
    }

    /**
     * @param source
     * @param geoPoint
     * @return
     */
    protected KDNode2 add(KDNode2 source, GeoPoint geoPoint) {
        if (source == null) {
            return new KDNode2(geoPoint);
        }

        final int cmp = order(source).compare(source.getCoordinate(), geoPoint);

        if (cmp <= 0) {

            source.setLeft(add(source.getLeft(), geoPoint));
            source.getLeft().setHeight(source.getHeight() + 1);
            source.getLeft().setParent(source);

        } else {

            source.setRight(add(source.getRight(), geoPoint));
            source.getRight().setHeight(source.getHeight() + 1);
            source.getRight().setParent(source);

        }

        return source;
    }

    /**
     * @param geoPoint
     */
    @Override
    public boolean add(GeoPoint geoPoint) {
        if (getRoot() == null) {
            setRoot(new KDNode2(geoPoint, 0));
            return true;
        }

        setRoot(add(getRoot(), geoPoint));

        return true;
    }

    /**
     * @param collection
     */
    @Override
    public boolean addAll(Collection<? extends GeoPoint> collection) {
        for ( GeoPoint each : collection ) {
            if (!add(each)) {
                return false;
            }
        }

        return true;
    }

    /**
     * @param source
     * @param quadrant
     * @param collector
     * @return
     */
    protected List<GeoPoint> query(KDNode2 source, Quadrant quadrant, List<GeoPoint> collector) {
        if (source == null) {
            return collector;
        }

        if (quadrant.contains(source.getCoordinate())) {

            collector.add(source.getCoordinate());

            collector = query(source.getLeft(), quadrant, collector);
            collector = query(source.getRight(), quadrant, collector);

        } else {
            final int cmp = order(source).compare(source.getCoordinate(), quadrant.getCenter());

            if (cmp <= 0) {
                collector = query(source.getLeft(), quadrant, collector);
            } else if (cmp > 0) {
                collector = query(source.getRight(), quadrant, collector);
            }
        }

        return collector;
    }

    /**
     * @param quadrant
     * @return
     */
    public List<GeoPoint> query(Quadrant quadrant) {
        final KDNode2 ref = getRoot();

        return query(ref, quadrant, new ArrayList<>());
    }

    /**
     * @param p0
     * @param p1
     * @return
     */
    public List<GeoPoint> query(GeoPoint p0, GeoPoint p1) {
        return query(new Quadrant(p0, p1));
    }

    /**
     * @param source
     * @param target
     * @param found
     * @param distance
     * @return
     */
    protected GeoPoint nearestNeighbor(KDNode2 source, GeoPoint target, GeoPoint found, double distance) {
        if (source == null) {
            return null;
        }

        double reference = distance;
        double tmp = distanceCalc.calc(source.getCoordinate(), target);

        if (tmp < distance) {
            found = source.getCoordinate();
            reference = tmp;
        }

        final int cmp = order(source).compare(source.getCoordinate(), target);
        if (cmp <= 0) {
            if (source.getLeft() == null) {
                return found;
            }

            return nearestNeighbor(source.getLeft(), target, found, reference);
        } else if (cmp > 0) {
            if (source.getRight() == null) {
                return found;
            }

            return nearestNeighbor(source.getRight(), target, found, reference);
        }

        return null;
    }

    /**
     * @param geoPoint
     * @return
     */
    public GeoPoint nearestNeighbor(GeoPoint geoPoint) {
        if (getRoot() == null) {
            return null;
        }

        return nearestNeighbor(getRoot(), geoPoint, null, Double.MAX_VALUE);
    }

    /**
     * @param src
     * @param collector
     * @return
     */
    protected List<GeoPoint> distinct(KDNode2 src, List<GeoPoint> collector) {
        if (src == null) {
            return collector;
        } else if (nonNull(src.getCoordinate())) {
            collector.add(src.getCoordinate());
        }

        collector = distinct(src.getLeft(), collector);
        collector = distinct(src.getRight(), collector);

        return collector;
    }

    /**
     * @return
     */
    public Enumeration<GeoPoint> enumeration() {
        final KDNode2 ref = getRoot();

        return new KDEnumerator(ref);
    }

    /**
     * @return
     */
    @Override
    public Iterator<GeoPoint> iterator() {
        final KDNode2 ref = getRoot();

        return distinct(ref, new ArrayList<>()).iterator();
    }

    /**
     * @return
     */
    public List<GeoPoint> list() {
        final KDNode2 ref = getRoot();

        return distinct(ref, new ArrayList<>());
    }

    /**
     * @param source
     * @param geoPoint
     * @return
     */
    protected KDNode2 getNode(KDNode2 source, GeoPoint geoPoint) {
        if (source == null) {
            return null;
        }

        final int cmp = order(source).compare(source.getCoordinate(), geoPoint);

        if (source.getCoordinate().equals(geoPoint)) {
            return source;
        } else if (cmp < 0) {
            return getNode(source.getLeft(), geoPoint);
        } else if (cmp > 0) {
            return getNode(source.getRight(), geoPoint);
        }

        return null;
    }

    /**
     * @param geoPoint
     * @return
     */
    public GeoPoint get(GeoPoint geoPoint) {
        if (getRoot() == null) {
            return null;
        }

        return getNode(getRoot(), geoPoint).getCoordinate();
    }

    /**
     * @param geoPoint
     * @return
     */
    @Override
    public boolean contains(Object geoPoint) {
        if (getRoot() == null) {
            return false;
        }

        return getNode(getRoot(), (GeoPoint) geoPoint) != null;
    }

    /**
     * @param collection
     * @return
     */
    @Override
    public boolean containsAll(Collection<?> collection) {
        for ( Object each : collection ) {
            if (!contains(each)) {
                return false;
            }
        }

        return true;
    }

    /**
     * @param node
     * @return
     */
    protected Comparator<GeoPoint> order(KDNode2 node) {
        if ((node.getHeight() % K) == 0) {
            return xComparator;
        } else if ((node.getHeight() % 2) == 1) {
            return yComparator;
        }

        throw new IllegalStateException("cannot happen");
    }

    /**
     * @param source
     * @return
     */
    protected int size(KDNode2 source) {
        if (source == null) {
            return 0;
        }

        return 1 + size(source.getLeft()) + size(source.getRight());
    }

    /**
     * @return
     */
    @Override
    public int size() {
        if (getRoot() == null) {
            return 0;
        }

        return size(getRoot());
    }

    /**
     * @return
     */
    @Override
    public boolean isEmpty() {
        return size() == 0;
    }

    @Override
    public Object[] toArray() {
        return distinct(getRoot(), new ArrayList<>()).toArray();
    }

    @Override
    public <T> T[] toArray(T[] a) {
        return distinct(getRoot(), new ArrayList<>()).toArray(a);
    }

    @Override
    public void clear() {
        setRoot(null);
    }

    @Override
    public boolean remove(Object o) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public boolean removeAll(Collection<?> c) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public boolean retainAll(Collection<?> c) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * @param source
     * @return
     */
    protected int height(KDNode2 source) {
        if (source == null) {
            return 0;
        }

        return 1 + Math.max(height(source.getLeft()), height(source.getRight()));
    }

    /**
     * @return
     */
    public int height() {
        if (getRoot() == null) {
            return 0;
        }

        return height(getRoot());
    }

    /**
     * @param root
     */
    public void setRoot(KDNode2 root) {
        this.root = root;
    }

    /**
     * @return
     */
    public KDNode2 getRoot() {
        return root;
    }

    /**
     * @return
     */
    public Comparator<GeoPoint> getxComparator() {
        return xComparator;
    }

    /**
     * @return
     */
    public Comparator<GeoPoint> getyComparator() {
        return yComparator;
    }

    /**
     * @return
     */
    public DistanceCalc getDistanceCalc() {
        return distanceCalc;
    }
}
