/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-algorithm project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.algorithm;

import terraml.algorithm.GeoPoint;
import java.io.Serializable;
import static java.lang.Math.max;
import static java.lang.Math.min;
import java.util.Objects;
import static terraml.commons.Doubles.isGreaterEqual;
import static terraml.commons.Doubles.isSmallerEqual;
import terraml.commons.tuple.Pair;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 * @param <Q>
 */
public class Quadrant implements Serializable {

    private String data;

    private GeoPoint sw;
    private GeoPoint ne;

    /**
     * @param data
     * @param sw
     * @param ne
     */
    public Quadrant(String data, GeoPoint sw, GeoPoint ne) {
        this.data = data;

        Pair<GeoPoint, GeoPoint> pair = boundsOf(sw, ne);

        this.sw = pair.key;
        this.ne = pair.value;
    }

    /**
     * @param sw
     * @param ne
     */
    public Quadrant(GeoPoint sw, GeoPoint ne) {
        this(null, sw, ne);
    }

    /**
     * @param x0
     * @param y0
     * @param x1
     * @param y1
     */
    public Quadrant(double x0, double y0, double x1, double y1) {
        this(new GeoPoint(x0, y0, 0), new GeoPoint(x1, y1, 0));
    }

    /**
     * @param data
     * @param x0
     * @param y0
     * @param x1
     * @param y1
     */
    public Quadrant(String data, double x0, double y0, double x1, double y1) {
        this(data, new GeoPoint(x0, y0, 0), new GeoPoint(x1, y1, 0));
    }

    /**
     * @param entries
     * @return
     */
    public static Pair<GeoPoint, GeoPoint> boundsOf(GeoPoint... entries) {
        double _latMin = Double.POSITIVE_INFINITY;
        double _lonMin = Double.POSITIVE_INFINITY;
        double _latMax = Double.NEGATIVE_INFINITY;
        double _lonMax = Double.NEGATIVE_INFINITY;

        for ( GeoPoint entry : entries ) {
            // zamansız öttü bir kesim,
            // zararlılarla değil işim,
            // sıkıldım yazdı sağ elim,
            // beynimde gizli mağbedim.
            _latMin = min(_latMin, entry.getX());
            _latMax = max(_latMax, entry.getX());
            _lonMin = min(_lonMin, entry.getY());
            _lonMax = max(_lonMax, entry.getY());
        }

        return new Pair<>(new GeoPoint(_latMin, _lonMin), new GeoPoint(_latMax, _lonMax));
    }

    /**
     * @param quadrant
     * @param p0
     * @param p1
     * @return
     */
    public static boolean contains(Quadrant quadrant, GeoPoint p0, GeoPoint p1) {
        Pair<GeoPoint, GeoPoint> pair = boundsOf(p0, p1);
        GeoPoint _sw = pair.key;
        GeoPoint _ne = pair.value;

        return isGreaterEqual(_sw.getX(), quadrant.getSw().getX())
                && isSmallerEqual(_ne.getX(), quadrant.getNe().getX())
                && isGreaterEqual(_sw.getY(), quadrant.getSw().getY())
                && isSmallerEqual(_ne.getY(), quadrant.getNe().getY());
    }

    /**
     * @param quadrant
     * @param p0
     * @return
     */
    public static boolean contains(Quadrant quadrant, GeoPoint p0) {
        return isGreaterEqual(p0.getX(), quadrant.getSw().getX())
                && isGreaterEqual(p0.getY(), quadrant.getSw().getY())
                && isSmallerEqual(p0.getX(), quadrant.getNe().getX())
                && isSmallerEqual(p0.getY(), quadrant.getNe().getY());
    }

    /**
     * @param quadrant
     * @param p0
     * @param p1
     * @return
     */
    public static boolean intersects(Quadrant quadrant, GeoPoint p0, GeoPoint p1) {
        Pair<GeoPoint, GeoPoint> pair = boundsOf(p0, p1);
        GeoPoint _sw = pair.key;
        GeoPoint _ne = pair.value;

        return isGreaterEqual(_ne.getX(), quadrant.getSw().getX())
                && isSmallerEqual(_sw.getX(), quadrant.getNe().getX())
                && isGreaterEqual(_ne.getY(), quadrant.getSw().getY())
                && isSmallerEqual(_sw.getY(), quadrant.getNe().getY());
    }

    /**
     * @param p0
     * @param p1
     * @return
     */
    public boolean intersects(GeoPoint p0, GeoPoint p1) {
        return intersects(this, p0, p1);
    }

    /**
     * @param quadrant
     * @return
     */
    public boolean intersects(Quadrant quadrant) {
        return intersects(this, quadrant.getSw(), getNe());
    }

    /**
     * @param quadrant
     * @return
     */
    public boolean contains(Quadrant quadrant) {
        return contains(this, quadrant.getSw(), quadrant.getNe());
    }

    /**
     * @param geoPoint
     * @return
     */
    public boolean contains(GeoPoint geoPoint) {
        return contains(this, geoPoint);
    }

    /**
     * @param sw
     * @param ne
     * @return
     */
    public boolean contains(GeoPoint p0, GeoPoint p1) {
        return contains(this, p0, p1);
    }

    /**
     * @return
     */
    public GeoPoint getCenter() {
        double lat = ((getNe().getX() - getSw().getX()) * 0.5) + getSw().getX();
        double lon = ((getNe().getY() - getSw().getY()) * 0.5) + getSw().getY();

        return new GeoPoint(lat, lon);
    }

    /**
     * @return
     */
    public GeoPoint getNw() {
        double lat = getNe().getX();
        double lon = getSw().getY();

        return new GeoPoint(lat, lon);
    }

    /**
     * @return
     */
    public GeoPoint getSe() {
        double lat = getSw().getX();
        double lon = getNe().getY();

        return new GeoPoint(lat, lon);
    }

    /**
     * @return
     */
    public String getData() {
        return data;
    }

    /**
     * @param data
     */
    public void setData(String data) {
        this.data = data;
    }

    /**
     * @return
     */
    public GeoPoint getSw() {
        return sw;
    }

    /**
     * @param sw
     */
    public void setSw(GeoPoint sw) {
        this.sw = sw;
    }

    /**
     * @return
     */
    public GeoPoint getNe() {
        return ne;
    }

    /**
     * @param ne
     */
    public void setNe(GeoPoint ne) {
        this.ne = ne;
    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 83 * hash + Objects.hashCode(this.data);
        hash = 83 * hash + Objects.hashCode(this.sw);
        hash = 83 * hash + Objects.hashCode(this.ne);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final Quadrant other = (Quadrant) obj;
        if (!Objects.equals(this.data, other.data)) {
            return false;
        }
        if (!Objects.equals(this.sw, other.sw)) {
            return false;
        }
        if (!Objects.equals(this.ne, other.ne)) {
            return false;
        }
        return true;
    }

}
