/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-algorithm project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.algorithm.node;

import java.io.Serializable;
import java.util.Map;
import java.util.Objects;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public class AStarNode<Q> implements Serializable {

    private final Q id;
    private final Map<Q, Double> heuristic;

    private double distanceFromSource;
    private double heuristicOfDestination;
    private double cost;

    /**
     * @param nodeId
     * @param heuristic
     */
    public AStarNode(Q nodeId, Map<Q, Double> heuristic) {
        this.id = nodeId;
        this.distanceFromSource = Double.MAX_VALUE;
        this.heuristic = heuristic;
    }

    /**
     * @return
     */
    public Q getId() {
        return id;
    }

    /**
     * @return
     */
    public double getDistanceFromSource() {
        return distanceFromSource;
    }

    /**
     * @param distanceFromSource
     */
    public void setDistanceFromSource(double distanceFromSource) {
        this.distanceFromSource = distanceFromSource;
    }

    /**
     * @param destination
     */
    public void calcF(Q destination) {
        this.heuristicOfDestination = heuristic.get(destination);
        this.cost = distanceFromSource + heuristicOfDestination;
    }

    /**
     * @return
     */
    public double getHeuristicOfDestination() {
        return heuristicOfDestination;
    }

    /**
     * @return
     */
    public double getCost() {
        return cost;
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 89 * hash + Objects.hashCode(this.id);
        hash = 89 * hash + Objects.hashCode(this.heuristic);
        hash = 89 * hash + (int) (Double.doubleToLongBits(this.distanceFromSource) ^ (Double.doubleToLongBits(this.distanceFromSource) >>> 32));
        hash = 89 * hash + (int) (Double.doubleToLongBits(this.heuristicOfDestination) ^ (Double.doubleToLongBits(this.heuristicOfDestination) >>> 32));
        hash = 89 * hash + (int) (Double.doubleToLongBits(this.cost) ^ (Double.doubleToLongBits(this.cost) >>> 32));
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final AStarNode<?> other = (AStarNode<?>) obj;
        if (Double.doubleToLongBits(this.distanceFromSource) != Double.doubleToLongBits(other.distanceFromSource)) {
            return false;
        }
        if (Double.doubleToLongBits(this.heuristicOfDestination) != Double.doubleToLongBits(other.heuristicOfDestination)) {
            return false;
        }
        if (Double.doubleToLongBits(this.cost) != Double.doubleToLongBits(other.cost)) {
            return false;
        }
        if (!Objects.equals(this.id, other.id)) {
            return false;
        }
        if (!Objects.equals(this.heuristic, other.heuristic)) {
            return false;
        }
        return true;
    }
}
