/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-algorithm project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.algorithm.node;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import static java.util.Objects.nonNull;
import static terraml.commons.Objects.isNull;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public class AVLNode<Q> implements Comparator<Q>, Serializable {

    private final Comparator<Q> comparator;

    // tutmuşlar iki yakamdan isminizi istiyorlar.
    private AVLNode<Q> parent;
    private AVLNode<Q> left;
    private AVLNode<Q> right;

    private Q data;
    private int height;
    private int balance;

    private List<Q> duplicates;

    /**
     *
     * @param comp
     * @param parent
     * @param data
     */
    public AVLNode(Comparator<Q> comp, AVLNode<Q> parent, Q data) {
        this.comparator = comp;
        this.parent = parent;
        this.data = data;
    }

    /**
     * @param comp
     * @param data
     */
    public AVLNode(Comparator<Q> comp, Q data) {
        this(comp, null, data);
    }

    /**
     * @return
     */
    public boolean hasDuplicates() {
        return nonNull(duplicates) && !duplicates.isEmpty();
    }

    /**
     * @param data
     * @return
     */
    public boolean addDuplicate(Q data) {
        if (isNull(duplicates)) {
            this.duplicates = new ArrayList<>();
        }

        return this.duplicates.add(data);
    }

    @Override
    public int compare(Q o1, Q o2) {
        return comparator.compare(o1, o2);
    }

    /**
     *
     * @return
     */
    public AVLNode<Q> getParent() {
        return parent;
    }

    /**
     *
     * @param parent
     */
    public void setParent(AVLNode<Q> parent) {
        this.parent = parent;
    }

    /**
     *
     * @return
     */
    public AVLNode<Q> getLeft() {
        return left;
    }

    /**
     *
     * @param left
     */
    public void setLeft(AVLNode<Q> left) {
        this.left = left;
    }

    /**
     *
     * @return
     */
    public AVLNode<Q> getRight() {
        return right;
    }

    /**
     *
     * @param right
     */
    public void setRight(AVLNode<Q> right) {
        this.right = right;
    }

    /**
     *
     * @return
     */
    public Q getData() {
        return data;
    }

    /**
     *
     * @param data
     */
    public void setData(Q data) {
        this.data = data;
    }

    /**
     *
     * @return
     */
    public int getHeight() {
        return height;
    }

    /**
     *
     * @param height
     */
    public void setHeight(int height) {
        this.height = height;
    }

    /**
     *
     * @return
     */
    public int getBalance() {
        return balance;
    }

    /**
     *
     * @param balance
     */
    public void setBalance(int balance) {
        this.balance = balance;
    }

    /**
     *
     * @return
     */
    public List<Q> getDuplicates() {
        return duplicates;
    }

    /**
     *
     * @param duplicates
     */
    public void setDuplicates(List<Q> duplicates) {
        this.duplicates = duplicates;
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 71 * hash + Objects.hashCode(this.comparator);
        hash = 71 * hash + Objects.hashCode(this.parent);
        hash = 71 * hash + Objects.hashCode(this.left);
        hash = 71 * hash + Objects.hashCode(this.right);
        hash = 71 * hash + Objects.hashCode(this.data);
        hash = 71 * hash + this.height;
        hash = 71 * hash + this.balance;
        hash = 71 * hash + Objects.hashCode(this.duplicates);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final AVLNode<?> other = (AVLNode<?>) obj;
        if (this.height != other.height) {
            return false;
        }
        if (this.balance != other.balance) {
            return false;
        }
        if (!Objects.equals(this.comparator, other.comparator)) {
            return false;
        }
        if (!Objects.equals(this.parent, other.parent)) {
            return false;
        }
        if (!Objects.equals(this.left, other.left)) {
            return false;
        }
        if (!Objects.equals(this.right, other.right)) {
            return false;
        }
        if (!Objects.equals(this.data, other.data)) {
            return false;
        }
        if (!Objects.equals(this.duplicates, other.duplicates)) {
            return false;
        }
        return true;
    }
}
