/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-algorithm project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.algorithm.node;

import java.io.Serializable;
import java.util.Objects;
import terraml.algorithm.GeoPoint;
import terraml.algorithm.Quadrant;
import static terraml.commons.Objects.isNull;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public class PQuadNode implements Serializable {

    public static enum TypeEnum {

        CELL,
        CELL_GROUP;
    }

    private Quadrant quadrant;
    private GeoPoint coordinate;

    private PQuadNode parent;
    private TypeEnum typeEnum = TypeEnum.CELL;

    private PQuadNode southWestNode;
    private PQuadNode northEastNode;
    private PQuadNode northWestNode;
    private PQuadNode southEastNode;

    public PQuadNode() {
    }

    /**
     * @param coordinate
     */
    public PQuadNode(GeoPoint coordinate) {
        this.coordinate = coordinate;
    }

    public void split() {
        setNorthEastNode(new PQuadNode());
        getNorthEastNode().setQuadrant(new Quadrant(getQuadrant().getCenter(), getQuadrant().getNe()));
        getNorthEastNode().setParent(this);

        setSouthEastNode(new PQuadNode());
        getSouthEastNode().setQuadrant(new Quadrant(getQuadrant().getCenter(), getQuadrant().getSe()));
        getSouthEastNode().setParent(this);

        setSouthWestNode(new PQuadNode());
        getSouthWestNode().setQuadrant(new Quadrant(getQuadrant().getCenter(), getQuadrant().getSw()));
        getSouthWestNode().setParent(this);

        setNorthWestNode(new PQuadNode());
        getNorthWestNode().setQuadrant(new Quadrant(getQuadrant().getCenter(), getQuadrant().getNw()));
        getNorthWestNode().setParent(this);

        setCoordinate(null);
        setTypeEnum(TypeEnum.CELL_GROUP);
    }

    /**
     * @return
     */
    public boolean isNorthWestEmpty() {
        return isNull(getNorthWestNode());
    }

    /**
     * @return
     */
    public boolean isNorthEastEmpty() {
        return isNull(getNorthEastNode());
    }

    /**
     * @return
     */
    public boolean isSouthEastEmpty() {
        return isNull(getSouthEastNode());
    }

    /**
     * @return
     */
    public boolean isSouthWestEmpty() {
        return isNull(getSouthWestNode());
    }

    /**
     * @return
     */
    public boolean isLeaf() {
        return isNorthWestEmpty()
                && isNorthEastEmpty()
                && isSouthEastEmpty()
                && isSouthWestEmpty();
    }

    /**
     * @return
     */
    public boolean isOrphan() {
        return isNull(getParent());
    }

    /**
     * @return
     */
    public Quadrant getQuadrant() {
        return quadrant;
    }

    /**
     * @param quadrant
     */
    public void setQuadrant(Quadrant quadrant) {
        this.quadrant = quadrant;
    }

    /**
     * @return
     */
    public GeoPoint getCoordinate() {
        return coordinate;
    }

    /**
     * @param coordinate
     */
    public void setCoordinate(GeoPoint coordinate) {
        this.coordinate = coordinate;
    }

    /**
     * @return
     */
    public PQuadNode getParent() {
        return parent;
    }

    /**
     * @param parent
     */
    public void setParent(PQuadNode parent) {
        this.parent = parent;
    }

    /**
     * @return
     */
    public TypeEnum getTypeEnum() {
        return typeEnum;
    }

    /**
     * @param typeEnum
     */
    public void setTypeEnum(TypeEnum typeEnum) {
        this.typeEnum = typeEnum;
    }

    /**
     * @return
     */
    public PQuadNode getSouthWestNode() {
        return southWestNode;
    }

    /**
     * @param southWestNode
     */
    public void setSouthWestNode(PQuadNode southWestNode) {
        this.southWestNode = southWestNode;
    }

    /**
     * @return
     */
    public PQuadNode getNorthEastNode() {
        return northEastNode;
    }

    /**
     * @param northEastNode
     */
    public void setNorthEastNode(PQuadNode northEastNode) {
        this.northEastNode = northEastNode;
    }

    /**
     * @return
     */
    public PQuadNode getNorthWestNode() {
        return northWestNode;
    }

    /**
     * @param northWestNode
     */
    public void setNorthWestNode(PQuadNode northWestNode) {
        this.northWestNode = northWestNode;
    }

    /**
     * @return
     */
    public PQuadNode getSouthEastNode() {
        return southEastNode;
    }

    /**
     * @param southEastNode
     */
    public void setSouthEastNode(PQuadNode southEastNode) {
        this.southEastNode = southEastNode;
    }

    @Override
    public int hashCode() {
        int hash = 3;
        hash = 11 * hash + Objects.hashCode(this.quadrant);
        hash = 11 * hash + Objects.hashCode(this.coordinate);
        hash = 11 * hash + Objects.hashCode(this.parent);
        hash = 11 * hash + Objects.hashCode(this.typeEnum);
        hash = 11 * hash + Objects.hashCode(this.southWestNode);
        hash = 11 * hash + Objects.hashCode(this.northEastNode);
        hash = 11 * hash + Objects.hashCode(this.northWestNode);
        hash = 11 * hash + Objects.hashCode(this.southEastNode);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final PQuadNode other = (PQuadNode) obj;
        if (!Objects.equals(this.quadrant, other.quadrant)) {
            return false;
        }
        if (!Objects.equals(this.coordinate, other.coordinate)) {
            return false;
        }
        if (!Objects.equals(this.parent, other.parent)) {
            return false;
        }
        if (this.typeEnum != other.typeEnum) {
            return false;
        }
        if (!Objects.equals(this.southWestNode, other.southWestNode)) {
            return false;
        }
        if (!Objects.equals(this.northEastNode, other.northEastNode)) {
            return false;
        }
        if (!Objects.equals(this.northWestNode, other.northWestNode)) {
            return false;
        }
        if (!Objects.equals(this.southEastNode, other.southEastNode)) {
            return false;
        }
        return true;
    }
}
