/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial;

import terraml.commons.annotation.Development;
import terraml.commons.annotation.File;
import terraml.commons.math.Angle;
import static terraml.geospatial.GeoUtils.initialBearingFromRadian;
import static terraml.geospatial.GeoUtils.lat2rad;
import static terraml.geospatial.GeoUtils.lon2rad;

// Bir kağıda adalet yaz, bir köşede yak.
/**
 * North Based Azimuth Schema:
 *
 * 0,360
 * 270 90
 * 180
 *
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@File(
        fileName = "Azimuths.java",
        packageName = "terraml.geospatial",
        projectName = "terraml-geospatial"
)
public final class Azimuths {

    private Azimuths() {
    }

    /**
     * @param double
     * @param double
     * @param double
     * @param double
     * @return
     */
    public static Angle northBasedAzimuthFromRadian(double lat0, double lon0, double lat1, double lon1) {
        return Angle.fromDegree((360 + initialBearingFromRadian(lat0, lon0, lat1, lon1)) % 360);
    }

    /**
     * 0,360
     * 270 90
     * 180
     *
     * @param Latlon
     * @param Latlon
     * @return Azimuth angle as Angle.
     */
    @Development(status = Development.Status.STABLE)
    public static Angle northBasedAzimuth(Latlon latlon0, Latlon latlon1) {
        return Angle.fromDegree((360 + initialBearingFromRadian(lat2rad(latlon0), lon2rad(latlon0), lat2rad(latlon1), lon2rad(latlon1))) % 360);
    }

    /**
     * @param double
     * @param double
     * @param double
     * @param double
     * @return
     */
    public static Angle northBasedFinalAzimuthFromRadian(double lat0, double lon0, double lat1, double lon1) {
        return Angle.fromDegree(((initialBearingFromRadian(lat0, lon0, lat1, lon1) + 180)) % 360);
    }

    /**
     * @param Latlon
     * @param Latlon
     * @return
     */
    public static Angle northBasedFinalAzimuth(Latlon latlon0, Latlon latlon1) {
        return Angle.fromDegree(((initialBearingFromRadian(lat2rad(latlon0), lon2rad(latlon0), lat2rad(latlon1), lon2rad(latlon1)) + 180) % 360));
    }
}
