/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial;

import java.util.Collection;
import terraml.commons.annotation.Development;
import terraml.commons.annotation.File;
import terraml.commons.annotation.Issue;
import terraml.commons.math.Angle;

// herkes delirmiş, hiç etkinlikler etik değil.
/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
@File(
        fileName = "GeoBoundingBox.java",
        packageName = "terraml.geospatial",
        projectName = "terraml-geospatial"
)
public interface GeoBoundingBox extends GeoShape {

    /**
     * @param coordinate
     * @return true if this geobox contains given Latlon. Otherwise false.
     */
    @Development(status = Development.Status.STABLE)
    public boolean contains(Latlon coordinate);

    /**
     * Calculation is not accuarate. It's compatible with Leaflet Bounds.
     *
     * @param coordinate
     * @return true if this geobox contains given geobox. Otherwise false.
     */
    @Issue(
            category = Issue.Type.ENHANCEMENT,
            content = "Everytime get latlon from arg, it creates a new object. (Immutability)",
            isFixed = true,
            issueId = "TMLSPA-3"
    )
    public boolean contains(GeoBoundingBox coordinate);

    /**
     *
     * Calculation is not accuarate. It's compatible with Leaflet Bounds.
     *
     * @param rectangle
     * @return true if this geobox intersects with given geobox. Otherwise false.
     */
    @Issue(
            category = Issue.Type.ENHANCEMENT,
            content = "Everytime get latlon from arg, it creates a new object. (Immutability)",
            isFixed = true,
            issueId = "TMLSPA-4"
    )
    public boolean intersects(GeoBoundingBox geoBox);

    /**
     * Since it's heuristic, it is not an accurate calculation. Do not use it for critical calculations.
     *
     * @return GeoSegment Collection fo GeoBounding Box
     */
    public Collection<GeoSegment> toHeuristicSegments();

    /**
     * @return throws OperationNotSupportedException.
     */
    public Collection<GeoSegment> toAccurateSegments();

    /**
     * @return A GeoPolyline converted from this GeoBoundingBox with Clock-Wised GeoSegments.
     */
    public GeoPolyline toPolylineCW();

    /**
     * @return A GeoPolyline converted from this GeoBoundingBox with Counter-Clock-Wised GeoSegments.
     */
    public GeoPolyline toPolylineCCW();

    /**
     * Not accurate but compatible with Leaflet.
     *
     * @see getCenterAccurate()
     * @return Center of GeoBoundingBox
     */
    @Development(status = Development.Status.STABLE)
    public Latlon getCenter();

    /**
     * getCenter() ile arasında 14km 3 metre hesaplama farkı oluşuyor.
     * <p>
     * Use it for accurate calculation.
     *
     * @return Center of GeoBoundingBox
     */
    @Development(status = Development.Status.STABLE)
    public Latlon getCenterAccurate();

    /**
     * @param distanceMeter
     * @param bearing
     * @return throws UnsupportedOperationException
     */
    public GeoBoundingBox offset(DistanceNode distance, Angle bearing);

    /**
     * @return
     */
    public Latlon getSouthWest();

    /**
     * @return
     */
    public Latlon getNorthEast();

    /**
     * @return
     */
    public Latlon[] toArray();

    /**
     * @return
     */
    public double[] toDoubleArray();

    @Override
    public GeoBoundingBox clone();

    @Override
    public GeoShapeUnit getGeoShapeUnit();

    @Override
    public Latlon[] getBounds();

    // takipteler ses etme sus.
}
