/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial;

import static java.lang.Math.atan2;
import static java.lang.Math.cos;
import static java.lang.Math.sin;
import static java.lang.Math.toDegrees;
import java.util.ArrayList;
import java.util.List;
import static terraml.commons.Doubles.isEqual;
import static terraml.commons.Doubles.isGreater;
import static terraml.commons.Doubles.isSmaller;
import terraml.commons.annotation.Development;
import terraml.commons.math.Angle;
import terraml.commons.unit.DirectionUnit;

// Ağır uykusundan uyanmayan günü kurtarır.
/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public final class GeoUtils {

    private GeoUtils() {
    }

    public static final double EARTH_DIAMETER = Double.valueOf("12756.274");
    public static final double EARTH_RADIUS_KM = 6371.0;
    public static final double EARTH_RADIUS_M = 6371000.0;

    /**
     *
     *
     * @param latlon
     * @return latitude as radion from given Latlon Object.
     */
    public static double lat2rad(Latlon latlon) {
        return latlon.getLatitude().toRadian();
    }

    /**
     *
     * @param lat0
     * @param lon0
     * @param lat1
     * @param lon1
     * @return Calculates initial bearing from given latitudes and longitudes as form of radian. (degree)
     */
    public static double initialBearingFromRadian(double lat0, double lon0, double lat1, double lon1) {
        final double _lat0 = lat0;
        final double _lat1 = lat1;

        final double _dlon = lon1 - lon0;

        final double _letY = sin(_dlon) * cos(_lat1);
        final double _letX = cos(_lat0) * sin(_lat1) - sin(_lat0) * cos(_lat1) * cos(_dlon);

        return toDegrees(atan2(_letY, _letX));
    }

    /**
     *
     * @param lat0
     * @param lon0
     * @param lat1
     * @param lon1
     * @return
     */
    public static Angle bearingCCW(double lat0, double lon0, double lat1, double lon1) {
        return Angle.fromDegree((360 - ((360 + initialBearingFromRadian(lat0, lon0, lat1, lon1)))) % 360);
    }

    /**
     *
     * @param latlon0
     * @param latlon1
     * @return
     */
    public static Angle bearingCCW(Latlon latlon0, Latlon latlon1) {
        return Angle.fromDegree((360 - (360 + initialBearingFromRadian(lat2rad(latlon0), lon2rad(latlon0), lat2rad(latlon1), lon2rad(latlon1)))) % 360);
    }

    /**
     *
     * @param latlon0
     * @param latlon1
     * @return
     */
    @Development(status = Development.Status.STABLE)
    public static DirectionUnit direction(Latlon latlon0, Latlon latlon1) {
        double rounded = round(Azimuths.northBasedAzimuth(latlon0, latlon1).degree);

        if (isEqual(rounded, 90.0d)) {
            return DirectionUnit.EAST;
        } else if (isEqual(rounded, 180.0d)) {
            return DirectionUnit.SOUTH;
        } else if (isEqual(rounded, 270.0d)) {
            return DirectionUnit.WEST;
        } else if (isEqual(rounded, 360.0d)) {
            return DirectionUnit.NORTH;
        } else if (isSmaller(rounded, 90.0d)) {
            return DirectionUnit.NORTH_EAST;
        } else if (isGreater(rounded, 90.0d) && isSmaller(rounded, 180.0d)) {
            return DirectionUnit.SOUTH_EAST;
        } else if (isGreater(rounded, 180.0d) && isSmaller(rounded, 270.0d)) {
            return DirectionUnit.SOUTH_WEST;
        } else if (isGreater(rounded, 270.0d) && isSmaller(rounded, 360.0d)) {
            return DirectionUnit.NORTH_WEST;
        }

        return null;
    }

    /**
     *
     * @param longitude
     * @return
     */
    public static double fixLongitudeFromDegree(final double longitude) {
        return ((longitude + 540) % 360) - 180;
    }

    /**
     *
     * @param latlon
     * @return
     */
    public static double lon2rad(Latlon latlon) {
        return latlon.getLongitude().toRadian();
    }

    /**
     *
     * @param latlon
     * @return
     */
    public static double lat2deg(Latlon latlon) {
        return latlon.getLatitude().toDegree();
    }

    /**
     *
     * @param latlon
     * @return
     */
    public static double lon2deg(Latlon latlon) {
        return latlon.getLongitude().toDegree();
    }

    /**
     *
     * @param meter
     * @return
     */
    public static double toKilometer(double meter) {
        return meter / 1000.0;
    }

    /**
     *
     * @param kilometer
     * @return
     */
    public static double toMeter(double kilometer) {
        return kilometer * 1000.0;
    }

    /**
     *
     * @param value
     * @return
     */
    public static double round(double value) {
        return Math.round(value * 1000.0) / 1000;
    }

    /**
     *
     * @param list
     * @return
     */
    public static List<Latlon> openFormOf(final List<Latlon> list) {
        if (!list.get(0).equals(list.get(list.size() - 1))) {
            return list;
        }

        final List<Latlon> _vertices = new ArrayList<>(list);
        _vertices.remove(_vertices.size() - 1);

        return _vertices;
    }
}
