/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Objects;
import java.util.function.BiFunction;
import terraml.commons.Ints;
import terraml.commons.Longs;
import terraml.commons.tuple.LatlonEntry;
import terraml.commons.unit.DirectionUnit;
import terraml.geospatial.impl.ImmutableGeoBoundingBox;
import terraml.geospatial.impl.ImmutableLatlon;

// ışıklarım yanardı.

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public final class Geohash {

    private static class CellInfo {

        public int precision;
        public double widthMeter;
        public double heightMeter;

        public CellInfo(int precision, double widthMeter, double heightMeter) {
            this.precision = precision;
            this.widthMeter = widthMeter;
            this.heightMeter = heightMeter;
        }
    }

    private static final HashMap<Integer, CellInfo> HASHMAP;

    static {

        HASHMAP = new HashMap<>(9);

        HASHMAP.put(9, new CellInfo(1, 4.77, 4.77));
        HASHMAP.put(8, new CellInfo(2, 38.2, 19.1));
        HASHMAP.put(7, new CellInfo(3, 153.0, 153.0));
        HASHMAP.put(6, new CellInfo(4, 1.22 * 1000, 0.61 * 1000));
        HASHMAP.put(5, new CellInfo(5, 4.89 * 1000, 4.89 * 1000));
        HASHMAP.put(4, new CellInfo(6, 39.1 * 1000, 19.5 * 1000));
        HASHMAP.put(3, new CellInfo(7, 156.0 * 1000, 156.0 * 1000));
        HASHMAP.put(2, new CellInfo(8, 1250.0 * 1000, 625.0 * 1000));
        HASHMAP.put(1, new CellInfo(9, 5000.0 * 1000, 5000.0 * 1000));

    }

    private static final char[] BIT32 = {
        '0', '1', '2', '3', '4', '5', '6', '7',
        '8', '9', 'b', 'c', 'd', 'e', 'f', 'g',
        'h', 'j', 'k', 'm', 'n', 'p', 'q', 'r',
        's', 't', 'u', 'v', 'w', 'x', 'y', 'z'
    };

    private Geohash() {
    }

    // fevkaladenin fevkine vardım.
    
    /**
     *
     * @param coordinate
     * @return
     */
    public static String encodeWith(Latlon latlon) {
        return encode(latlon, 9);
    }

    /**
     *
     * @param rectangle
     * @return
     */
    public static String encodeWith(GeoBoundingBox rect) {
        
        Latlon _se = new ImmutableLatlon(rect.getSouthWest().lat(), rect.getNorthEast().lon());
        
        final double _w = Distance.vincenty(rect.getSouthWest(), _se);
        final double _h = Distance.vincenty(_se, rect.getNorthEast());

        for ( int _precision = 9; _precision >= 1; _precision-- ) {
            final CellInfo _cellInfo = HASHMAP.get(_precision);

            if ( isFit(_w, _h, _cellInfo.widthMeter, _cellInfo.heightMeter) ) {
                
                Latlon _center = Locate.centerOfBounds(rect.getSouthWest(), rect.getNorthEast());
                
                return encode(_center, _precision);
            }

        }

        return "";
    }

    /**
     *
     * @param w
     * @param h
     * @param lo
     * @param hi
     * @return
     */
    private static boolean isFit(final double w, final double h, final double lo, final double hi) {
        return ( w <= lo ) && ( h <= hi );
    }

    /**
     *
     * @param circle
     * @return
     */
    public static String encodeWith(GeoCircle circle) {
        return encodeWith(new ImmutableGeoBoundingBox(circle.getBounds()));
    }

    /**
     *
     * @param polyline
     * @return
     */
    public static String encodeWith(GeoPolyline polyline) {
        return encodeWith(new ImmutableGeoBoundingBox(polyline.getBounds()));
    }

    /**
     *
     * @param polygon
     * @return
     */
    public static String encodeWith(GeoPolygon polygon) {
        return encodeWith(new ImmutableGeoBoundingBox(polygon.getBounds()));
    }

    /**
     *
     * @return
     */
    private static BiFunction<DirectionUnit, Integer, String> findNeighbour() {
        return (DirectionUnit dir, Integer id) -> {
            if ( dir.equals(DirectionUnit.NORTH) ) {
                if ( id == 0 ) {
                    return "p0r21436x8zb9dcf5h7kjnmqesgutwvy";
                } else if ( id == 1 ) {
                    return "bc01fg45238967deuvhjyznpkmstqrwx";
                }
            } else if ( dir.equals(DirectionUnit.SOUTH) ) {
                if ( id == 0 ) {
                    return "14365h7k9dcfesgujnmqp0r2twvyx8zb";
                } else if ( id == 1 ) {
                    return "238967debc01fg45kmstqrwxuvhjyznp";
                }
            } else if ( dir.equals(DirectionUnit.EAST) ) {
                if ( id == 0 ) {
                    return "bc01fg45238967deuvhjyznpkmstqrwx";
                } else if ( id == 1 ) {
                    return "p0r21436x8zb9dcf5h7kjnmqesgutwvy";
                }
            } else if ( dir.equals(DirectionUnit.WEST) ) {
                if ( id == 0 ) {
                    return "238967debc01fg45kmstqrwxuvhjyznp";
                } else if ( id == 1 ) {
                    return "14365h7k9dcfesgujnmqp0r2twvyx8zb";
                }
            }

            return "";
        };
    }

    /**
     *
     * @return
     */
    private static BiFunction<DirectionUnit, Integer, String> findBorder() {
        return (DirectionUnit dir, Integer id) -> {
            if ( dir.equals(DirectionUnit.NORTH) ) {
                if ( id == 0 ) {
                    return "prxz";
                } else if ( id == 1 ) {
                    return "bcfguvyz";
                }
            } else if ( dir.equals(DirectionUnit.SOUTH) ) {
                if ( id == 0 ) {
                    return "028b";
                } else if ( id == 1 ) {
                    return "0145hjnp";
                }
            } else if ( dir.equals(DirectionUnit.EAST) ) {
                if ( id == 0 ) {
                    return "bcfguvyz";
                } else if ( id == 1 ) {
                    return "prxz";
                }
            } else if ( dir.equals(DirectionUnit.WEST) ) {
                if ( id == 0 ) {
                    return "0145hjnp";
                } else if ( id == 1 ) {
                    return "028b";
                }
            }

            return "";
        };
    }

    /**
     *
     * @param geohash
     * @param direction
     * @return
     */
    private static String _neighbour(final String geohash, DirectionUnit direction) {
        if ( Objects.isNull(geohash) || geohash.isEmpty() ) {
            return null;
        }

        String _last = geohash.substring(geohash.length() - 1);
        String _parent = geohash.substring(0, geohash.length() - 1);

        int _id = geohash.length() % 2;

        String _output = findBorder().apply(direction, _id);

        if ( _output.contains(_last) && !_parent.isEmpty() ) {
            _parent = _neighbour(_parent, direction);
        }

        String _code = findNeighbour().apply(direction, _id);
        int _index = _code.indexOf(_last);

        return _parent + BIT32[_index];
    }

    /**
     *
     * @param geohash
     * @return
     */
    public static Iterator<String> iterateAll(final String geohash) {
        LinkedList<String> linkedList = new LinkedList<>();

        linkedList.add(neighbour(geohash, DirectionUnit.NORTH));
        linkedList.add(neighbour(geohash, DirectionUnit.SOUTH));
        linkedList.add(neighbour(geohash, DirectionUnit.WEST));
        linkedList.add(neighbour(geohash, DirectionUnit.EAST));
        linkedList.add(neighbour(geohash, DirectionUnit.NORTH_WEST));
        linkedList.add(neighbour(geohash, DirectionUnit.NORTH_EAST));
        linkedList.add(neighbour(geohash, DirectionUnit.SOUTH_WEST));
        linkedList.add(neighbour(geohash, DirectionUnit.SOUTH_EAST));

        return linkedList.iterator();
    }

    /**
     *
     * @param geohash
     * @param direction
     * @return
     */
    public static String neighbour(final String geohash, DirectionUnit direction) {
        if ( direction.equals(DirectionUnit.NORTH) ) {
            return _neighbour(geohash, DirectionUnit.NORTH);
        } else if ( direction.equals(DirectionUnit.SOUTH) ) {
            return _neighbour(geohash, DirectionUnit.SOUTH);
        } else if ( direction.equals(DirectionUnit.WEST) ) {
            return _neighbour(geohash, DirectionUnit.WEST);
        } else if ( direction.equals(DirectionUnit.EAST) ) {
            return _neighbour(geohash, DirectionUnit.EAST);
        } else if ( direction.equals(DirectionUnit.NORTH_WEST) ) {
            return _neighbour(_neighbour(geohash, DirectionUnit.NORTH), DirectionUnit.WEST);
        } else if ( direction.equals(DirectionUnit.NORTH_EAST) ) {
            return _neighbour(_neighbour(geohash, DirectionUnit.NORTH), DirectionUnit.EAST);
        } else if ( direction.equals(DirectionUnit.SOUTH_WEST) ) {
            return _neighbour(_neighbour(geohash, DirectionUnit.SOUTH), DirectionUnit.WEST);
        } else if ( direction.equals(DirectionUnit.SOUTH_EAST) ) {
            return _neighbour(_neighbour(geohash, DirectionUnit.SOUTH), DirectionUnit.EAST);
        }

        return "";
    }
    /**
     *
     * @param entry
     * @param precision
     * @return
     */
    public static String encode(LatlonEntry entry, double precision) {
        int _index = 0, _currentBit = 0;
        boolean _isEven = true;
        String output = "";

        double _longitudeMin = Longitude.MIN_LONGITUDE, _longitudeMax = Longitude.MAX_LONGITUDE;
        double _latitudeMin = Latitude.MIN_LATITUDE, _latitudeMax = Latitude.MAX_LATITUDE;

        while ( output.length() < precision ) {

            if ( _isEven ) {
                double _midOf = ( _longitudeMin + _longitudeMax ) / 2;

                if ( entry.lon() >= _midOf ) {
                    _index = _index * 2 + 1;
                    _longitudeMin = _midOf;
                } else {
                    _index *= 2;
                    _longitudeMax = _midOf;
                }
            } else {
                double _midOf = ( _latitudeMin + _latitudeMax ) / 2;

                if ( entry.lat() >= _midOf ) {
                    _index = _index * 2 + 1;
                    _latitudeMin = _midOf;
                } else {
                    _index *= 2;
                    _latitudeMax = _midOf;
                }
            }
            _isEven = !_isEven;

            if ( ++_currentBit == 5 ) {
                output += BIT32[_index];
                _currentBit = 0;
                _index = 0;
            }

        }

        return output;
    }

    /**
     *
     * @param geohash
     * @return
     */
    public static LatlonEntry decode(final String geohash) {
        if ( Objects.isNull(geohash) || geohash.isEmpty() ) {
            return null;
        }

        GeoBoundry boundry = boundsOf(geohash);

        double _latMid = ( boundry.lowerBound.lat() + boundry.upperBound.lat() ) / 2;
        double _lonMid = ( boundry.lowerBound.lon() + boundry.upperBound.lon() ) / 2;

        //_latMid = Math.floor(2 - Math.log(boundry.upperBound.lat() - boundry.lowerBound.lat()) / Math.log(10));
        //_lonMid = Math.floor(2 - Math.log(boundry.upperBound.lon() - boundry.lowerBound.lon()) / Math.log(10));

        return new ImmutableLatlon(_latMid, _lonMid);
    }

    /**
     *
     * @param geohash
     * @return
     */
    public static GeoBoundry boundsOf(final String geohash) {
        if ( Objects.isNull(geohash) || geohash.isEmpty() ) {
            return null;
        }

        boolean _isEven = true;
        double _longitudeMin = Longitude.MIN_LONGITUDE, _longitudeMax = Longitude.MAX_LONGITUDE;
        double _latitudeMin = Latitude.MIN_LATITUDE, _latitudeMax = Latitude.MAX_LATITUDE;

        for ( int i = 0; i < geohash.length(); i++ ) {
            char _currentChar = geohash.charAt(i);
            int _index = indexOf(_currentChar);

            if ( Ints.isEqual(_index, -99) ) {
                return null;
            }

            for ( int j = 4; j >= 0; j-- ) {
                int bitN = _index >> j & 1;

                if ( _isEven ) {
                    double _midOf = ( _longitudeMin + _longitudeMax ) / 2;

                    if ( Longs.isEqual(bitN, 1) ) {
                        _longitudeMin = _midOf;
                    } else {
                        _longitudeMax = _midOf;
                    }
                } else {
                    double _midOf = ( _latitudeMin + _latitudeMax ) / 2;

                    if ( Longs.isEqual(bitN, 1) ) {
                        _latitudeMin = _midOf;
                    } else {
                        _latitudeMax = _midOf;
                    }
                }

                _isEven = !_isEven;

            }

        }

        return new GeoBoundry(new ImmutableLatlon(_latitudeMin, _longitudeMin), new ImmutableLatlon(_latitudeMax, _longitudeMax));
    }

    /**
     *
     * @param character
     * @return
     */
    private static int indexOf(final char character) {
        for ( int i = 0; i < BIT32.length; i++ ) {
            if ( BIT32[i] == character ) {
                return i;
            }
        }

        return -99;
    }
}