/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial;

import static java.lang.Math.atan2;
import static java.lang.Math.cos;
import static java.lang.Math.max;
import static java.lang.Math.min;
import static java.lang.Math.sin;
import static java.lang.Math.sqrt;
import static java.lang.Math.toDegrees;
import static java.lang.Math.toRadians;
import java.util.Arrays;
import java.util.List;
import terraml.commons.Ints;
import terraml.commons.annotation.Development;
import terraml.commons.tuple.LatlonEntry;
import static terraml.geospatial.GeoUtils.fixLongitudeFromDegree;
import static terraml.geospatial.GeoUtils.lat2deg;
import static terraml.geospatial.GeoUtils.lon2deg;
import terraml.geospatial.impl.ImmutableLatlon;

// Ağzımızda ateşten canavar sandığın cuğara
/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public final class Locate {

    private Locate() {
    }

    /**
     * @param lat0       lat in radian
     * @param lon0       lon in radian
     * @param lat1       lat in radian
     * @param lon1       lon in radian
     * @param percentage 10 ~ %10 like
     * @return lat, lon in degree
     */
    public static double[] locateWithPercentFromRadian(double lat0, double lon0, double lat1, double lon1, double percentage) {
        final double fraction = percentage / 10.0d;
        final double p0xSin = Math.sin(lat0);
        final double p0xCos = Math.cos(lat0);
        final double p0ySin = Math.sin(lon0);
        final double p0yCos = Math.cos(lon0);

        final double p1xSin = Math.sin(lat1);
        final double p1xCos = Math.cos(lat1);
        final double p1ySin = Math.sin(lon1);
        final double p1yCos = Math.cos(lon1);

        final double dx = lat1 - lat0;
        final double dy = lon1 - lon0;

        double Qx = Math.sin(dx / 2) * Math.sin(dx / 2);
        double Qy = Math.sin(dy / 2) * Math.sin(dy / 2);
        double Qt = Qx + Math.cos(lat0) * Math.cos(lat1) * Qy;
        double Qr = 2 * Math.atan2(Math.sqrt(Qt), Math.sqrt(1 - Qt));

        double Wt = Math.sin((1 - fraction) * Qr) / Math.sin(Qr);
        double Wr = Math.sin(fraction * Qr) / Math.sin(Qr);
        double Wx = Wt * p0xCos * p0yCos + Wr * p1xCos * p1yCos;
        double Wy = Wt * p0xCos * p0ySin + Wr * p1xCos * p1ySin;
        double Wz = Wt * p0xSin + Wr * p1xSin;

        double lat = Math.atan2(Wz, Math.sqrt(Wx * Wx + Wy * Wy));
        double lon = Math.atan2(Wy, Wx);

        lat = toDegrees(lat);
        lon = fixLongitudeFromDegree(toDegrees(lon));

        return new double[]{lat, lon};
    }

    /**
     * @param source     lat,lon in radian
     * @param target     lat, lon in radian
     * @param percentage 20 ~ %20 like
     * @return lat, lon in degrees.
     */
    public static double[] locateWithPercentFromRadian(double[] source, double[] target, double percentage) {
        return Locate.locateWithPercentFromRadian(source[0], source[1], target[0], target[1], percentage);
    }

    /**
     *
     * @param source
     * @param target
     * @param percentage
     * @return
     */
    public static Latlon locateWith(Latlon source, Latlon target, double percentage) {
        final double[] latlon = locateWithPercentFromRadian(source.toArrayAsRadian(), target.toArrayAsRadian(), percentage);

        return new ImmutableLatlon(latlon[0], latlon[1]);
    }

    /**
     *
     * @param lat0 lat in radian
     * @param lon0 lon in radian
     * @param lat1 lat in radian
     * @param lon1 lon in radian
     * @return lat, lon in degree
     */
    public static double[] halfWayFromRadian(double lat0, double lon0, double lat1, double lon1) {
        final double dy = toRadians(toDegrees(lon1) - toDegrees(lon0));
        final double Qx = cos(lat1) * cos(dy);
        final double Qy = cos(lat1) * sin(dy);
        final double Qt = (cos(lat0) + Qx);

        final double Wx = sqrt(Qt * Qt + Qy * Qy);
        final double Wy = sin(lat0) + sin(lat1);

        double lat = atan2(Wy, Wx);
        double lon = lon0 + atan2(Qy, Qt);

        lat = toDegrees(lat);
        lon = fixLongitudeFromDegree(toDegrees(lon));

        return new double[]{lat, lon};
    }

    /**
     * @param source
     * @param target
     * @return
     */
    public static double[] halfWayFromRadian(double[] source, double[] target) {
        return halfWayFromRadian(source[0], source[1], target[0], target[1]);
    }

    /**
     * @param source
     * @param target
     * @return
     */
    public static Latlon halfWay(Latlon source, Latlon target) {
        final double[] latlon = halfWayFromRadian(source.toArrayAsRadian(), target.toArrayAsRadian());

        return new ImmutableLatlon(latlon[0], latlon[1]);
    }

    /**
     * @param latlonList
     * @return
     */
    @Development(status = Development.Status.STABLE)
    public static <Q extends Latlon> Latlon centerOf(List<Q> latlonList) {
        final int _len = latlonList.size();

        if (Ints.isEqual(_len, 0)) {
            return null;
        }

        GeoVector _tmpGeoVector = new GeoVector(0, 0, 0);
        for ( Latlon _each : latlonList ) {
            _tmpGeoVector = _tmpGeoVector.translate(GeoVector.fromLatlon(_each));
        }

        double scalar = _len;

        return _tmpGeoVector.scale(1 / scalar).toLatlon();
    }

    /**
     * @param latlons
     * @return
     */
    public static <Q extends Latlon> List<Latlon> locateBoundsOf(List<Q> latlons) {
        double _latMin = Double.POSITIVE_INFINITY;
        double _lonMin = Double.POSITIVE_INFINITY;
        double _latMax = Double.NEGATIVE_INFINITY;
        double _lonMax = Double.NEGATIVE_INFINITY;

        for ( int i = 0; i < latlons.size(); i++ ) {
            _latMin = min(_latMin, lat2deg(latlons.get(i)));
            _latMax = max(_latMax, lat2deg(latlons.get(i)));
            _lonMin = min(_lonMin, lon2deg(latlons.get(i)));
            _lonMax = max(_lonMax, lon2deg(latlons.get(i)));
        }

        return Arrays.asList(new ImmutableLatlon(_latMin, _lonMin), new ImmutableLatlon(_latMax, _lonMax));
    }

    /**
     * @param <Q>
     * @param list
     * @return
     */
    public static <Q extends LatlonEntry> GeoBoundry boundsOf(List<Q> list) {
        double _latMin = Double.POSITIVE_INFINITY;
        double _lonMin = Double.POSITIVE_INFINITY;
        double _latMax = Double.NEGATIVE_INFINITY;
        double _lonMax = Double.NEGATIVE_INFINITY;

        for ( int i = 0; i < list.size(); i++ ) {
            _latMin = min(_latMin, list.get(i).lat());
            _latMax = max(_latMax, list.get(i).lat());
            _lonMin = min(_lonMin, list.get(i).lon());
            _lonMax = max(_lonMax, list.get(i).lon());
        }

        return new GeoBoundry(new ImmutableLatlon(_latMin, _lonMin), new ImmutableLatlon(_latMax, _lonMax));
    }

    /**
     * @param latlons
     * @return
     */
    public static Latlon[] locateBoundsOf(Latlon[] latlons) {
        double _latMin = Double.POSITIVE_INFINITY;
        double _lonMin = Double.POSITIVE_INFINITY;
        double _latMax = Double.NEGATIVE_INFINITY;
        double _lonMax = Double.NEGATIVE_INFINITY;

        for ( Latlon latlon : latlons ) {
            _latMin = min(_latMin, lat2deg(latlon));
            _latMax = max(_latMax, lat2deg(latlon));
            _lonMin = min(_lonMin, lon2deg(latlon));
            _lonMax = max(_lonMax, lon2deg(latlon));
        }

        return new Latlon[]{new ImmutableLatlon(_latMin, _lonMin), new ImmutableLatlon(_latMax, _lonMax)};
    }

    /**
     * @param entries
     * @return
     */
    public static GeoBoundry boundsOf(LatlonEntry... entries) {
        double _latMin = Double.POSITIVE_INFINITY;
        double _lonMin = Double.POSITIVE_INFINITY;
        double _latMax = Double.NEGATIVE_INFINITY;
        double _lonMax = Double.NEGATIVE_INFINITY;

        for ( LatlonEntry entry : entries ) {
            _latMin = min(_latMin, entry.lat());
            _latMax = max(_latMax, entry.lat());
            _lonMin = min(_lonMin, entry.lon());
            _lonMax = max(_lonMax, entry.lon());
        }

        return new GeoBoundry(new ImmutableLatlon(_latMin, _lonMin), new ImmutableLatlon(_latMax, _lonMax));
    }

    /**
     * ONLY FOR VERY VERY SMALL AREAS. If you need accurate result use:
     * Locate::halfway
     *
     * @param lowerBound
     * @param upperBound
     * @return
     */
    public static Latlon centerOfBounds(Latlon lowerBound, Latlon upperBound) {
        return new ImmutableLatlon(
                ((lat2deg(upperBound) - lat2deg(lowerBound)) * 0.5d) + lat2deg(lowerBound),
                ((lon2deg(upperBound) - lon2deg(lowerBound)) * 0.5d) + lon2deg(lowerBound)
        );
    }
}
