/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial;

import java.util.Collection;
import java.util.List;
import terraml.commons.tuple.Pair;
import terraml.geospatial.impl.ImmutableGeoBoundingBox;

// kepi çıkartıp kaskı deneyin.

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 * 
 */
public final class Overlap {

    private Overlap() {
    }
    
    /**
     * @param geoPolygon
     * @param polyline
     * @return true if given geoPolygon contains given GeoPolyline. Otherwise false.
     */
    public static boolean contains(GeoPolygon geoPolygon, GeoPolyline polyline) {
        final GeoBoundingBox _selfRect = new ImmutableGeoBoundingBox(geoPolygon.getBounds());
        final GeoBoundingBox _polyRect = new ImmutableGeoBoundingBox(polyline.getBounds());

        // do they share area
        if (!_selfRect.contains(_polyRect) || !_selfRect.intersects(_polyRect)) {
            return false;
        }

        // export wpoints of polyline
        Collection<Latlon> _ptr = polyline.toList();

        // is every wpoint of polyline inside
        for ( Latlon curr : _ptr ) {

            if (!geoPolygon.contains(curr)) {
                return false;
            }

        }

        // polygon might be concave
        // export segments of both
        final Collection<GeoSegment> _selfSeg = geoPolygon.toSegments();
        final Collection<GeoSegment> _polySeg = polyline.toSegments();

        // check for any intersection between segments of both
        if (Intersect.any(_selfSeg, _polySeg)) {
            return false;
        }

        return true;
    }
    
    /**
     * @param geoPolygon
     * @param rectangle
     * @return true if given geoPolygon contains given GeoBoundingBox. Otherwise false.
     */
    public static boolean contains(GeoPolygon geoPolygon, GeoBoundingBox rectangle) {
        final GeoBoundingBox _selfRect = new ImmutableGeoBoundingBox(geoPolygon.getBounds());

        // do they share some space
        if (!_selfRect.contains(rectangle) || !_selfRect.intersects(rectangle)) {
            return false;
        }

        // is every bound of rectangle inside
        for ( Latlon curr : rectangle.getBounds() ) {

            // if any bound is outside, then false.
            if (!geoPolygon.contains(curr)) {
                return false;
            }
        }

        // polygon might be concave
        // export segments of both
        final Collection<GeoSegment> _selfSeg = geoPolygon.toSegments();
        final Collection<GeoSegment> _rectSeg = rectangle.toHeuristicSegments();

        // check for any intersection between segments of both
        if (Intersect.any(_selfSeg, _rectSeg)) {
            return false;
        }

        return true;
    }
    
    /**
     * @param geoPolygon
     * @param circle
     * @return true if given geoPolygon contains given circle. Otherwise false.
     */
    public static boolean contains(GeoPolygon geoPolygon, GeoCircle circle) {
        // is center of circle inside
        if (!geoPolygon.contains(circle.getCenter())) {
            return false;
        }

        // convert both to rectangle
        final GeoBoundingBox _selfRect = new ImmutableGeoBoundingBox(geoPolygon.getBounds());
        final GeoBoundingBox _crclRect = new ImmutableGeoBoundingBox(circle.getBounds());

        // do they share area
        if (Intersect.disjoint(_selfRect, _crclRect)) {
            return false;
        }

        // is every bound of circle inside
        for ( Latlon curr : _crclRect.getBounds() ) {
            if (!geoPolygon.contains(curr)) {
                return false;
            }
        }

        // for certain result.
        // export both segmetnts
        final Collection<GeoSegment> _selfSeg = geoPolygon.toSegments();
        final Collection<GeoSegment> _crclSeg = _crclRect.toHeuristicSegments();

        // determine any existing intersection
        if (Intersect.any(_selfSeg, _crclSeg)) {
            return false;
        }

        return true;
    }
    
    /**
     * @param geoBox
     * @param polygon
     * @return true if given geobox contains given polygon. Otherwise false.
     */
    public static boolean contains(GeoBoundingBox geoBox, GeoPolygon polygon) {
        final GeoBoundingBox _polyRect = new ImmutableGeoBoundingBox(polygon.getBounds());

        // determine if this and given polyline shares area.
        if (!geoBox.contains(_polyRect) || !geoBox.intersects(_polyRect)) {
            return false;
        }

        // export polygon vertices
        Collection<Latlon> _ptr = polygon.getVertices();

        // is every wpoint of polygon inside this.
        for ( Latlon curr : _ptr ) {

            // if any wpoint is outside, then false.
            if (!geoBox.contains(curr)) {
                return false;
            }

        }

        // for certain result
        // export segments of this
        final Collection<GeoSegment> _selfSeg = geoBox.toHeuristicSegments();

        // export given polygon segments
        final Collection<GeoSegment> _polySeg = polygon.toSegments();

        // determine any intersection exist between segments
        for ( GeoSegment curr : _selfSeg ) {
            for ( GeoSegment next : _polySeg ) {

                // if any intersection exists, then false
                if (curr.intersects(next)) {
                    return false;
                }

            }

        }

        return true;
    }
    
    /**
     * @param geoBox
     * @param polyline
     * @return true if given geobox contains given polyline. Otherwise false.
     */
    public static boolean contains(GeoBoundingBox geoBox, GeoPolyline polyline) {
        final GeoBoundingBox _polyRect = new ImmutableGeoBoundingBox(polyline.getBounds());

        // determine if this and given polyline shares area.
        if (!geoBox.contains(_polyRect) || !geoBox.intersects(_polyRect)) {
            return false;
        }

        // export polylines wpoints
        final Collection<Latlon> _ptr = polyline.toList();

        // is every wpoint of polyline inside this.
        for ( Latlon curr : _ptr ) {

            // if any wpoint is outside, then false.
            if (!geoBox.contains(curr)) {
                return false;
            }
        }

        // for certain result
        // convert this to polyline
        final GeoPolyline _selfPol = geoBox.toPolylineCW();

        // if any intersection exist, then false.
        if (_selfPol.intersects(polyline)) {
            return false;
        }

        return true;
    }
    
    /**
     * @param geoBox
     * @param circle
     * @return true if given geobox contains given geocircle. Otherwise false.
     */
    public static boolean contains(GeoBoundingBox geoBox, GeoCircle circle) {
        final GeoBoundingBox _crclRect = new ImmutableGeoBoundingBox(circle.getBounds());

        // if given circle's center is outside of this. then false.
        if (!geoBox.contains(circle.getCenter())) {
            return false;
        }

        // determine if this and given circle shares area.
        if (!geoBox.contains(_crclRect) || !geoBox.intersects(_crclRect)) {
            return false;
        }

        // convert this to polyline
        GeoPolyline _selfPol = geoBox.toPolylineCW();

        // determine closest point to circle center
        Pair<Latlon, DistanceNode> _entry = _selfPol.closestOf(circle.getCenter(), new Distance.Vincenty());

        // if found closest point is inside rectangle, then it's not entirely contained
        if (circle.contains(_entry.getKey())) {
            return false;
        }

        return true;
    }
    
    /**
     * @param geoCircle
     * @param rectangle
     * @return true if given geocircle contains given geobox. Otherwise false.
     */
    public static boolean contains(GeoCircle geoCircle, GeoBoundingBox rectangle) {
        final GeoBoundingBox _selfRect = new ImmutableGeoBoundingBox(geoCircle.getBounds());

        // is rectangle inside bounds of this
        if (!_selfRect.contains(rectangle)) {
            return false;
        }

        // is every bound of given rectangle inside this.
        for ( Latlon curr : rectangle.getBounds() ) {

            // if any of bounds outside, then false.
            if (!geoCircle.contains(curr)) {
                return false;
            }
        }

        return true;
    }
    
    /**
     * @param geoCircle
     * @param polyline
     * @return true if given geocircle contains given polyline. Otherwise false.
     */
    public static boolean contains(GeoCircle geoCircle, GeoPolyline polyline) {
        final GeoBoundingBox _selfRect = new ImmutableGeoBoundingBox(geoCircle.getBounds());
        final GeoPolyline _selfPol = _selfRect.toPolylineCW();
        
        // avoid multiple construct.
        final List<Latlon> polyVertx = polyline.toList();
        
        final GeoBoundingBox _polyRect = new ImmutableGeoBoundingBox(polyVertx);

        // determine if given polyline and this shares area.
        if (!_selfRect.contains(_polyRect) || !_selfRect.intersects(_polyRect)) {
            return false;
        }

        final Collection<Latlon> _ptr = polyVertx;

        // is every wpoint contained.
        for ( Latlon curr : _ptr ) {

            // if any point of polyline is outside, then false.
            if (!geoCircle.contains(curr)) {
                return false;
            }

        }

        // determine if given polyline's segments overflow
        if (_selfPol.intersects(polyline)) {
            return false;
        }

        return true;
    }
    
    /**
     * @param geoCircle
     * @param polygon
     * @return true if geocircle contains geopolygon. False otherwise.
     */
    public static boolean contains(GeoCircle geoCircle, GeoPolygon polygon) {
        final GeoBoundingBox _selfRect = new ImmutableGeoBoundingBox(geoCircle.getBounds());
        final GeoBoundingBox _polyRect = new ImmutableGeoBoundingBox(polygon.getBounds());

        // determine if given polygon and this shares area.
        if (!_selfRect.contains(_polyRect) || !_selfRect.intersects(_polyRect)) {
            return false;
        }

        final Collection<Latlon> _ptr = polygon.getVertices();

        // is every wpoint contained.
        for ( Latlon curr : _ptr ) {

            // if any point of polyline is outside, then false.
            if (!geoCircle.contains(curr)) {
                return false;
            }

        }

        // for certain result
        final Collection<GeoSegment> _selfSeg = _selfRect.toHeuristicSegments();
        final Collection<GeoSegment> _polySeg = polygon.toSegments();

        // for every segment of this
        for ( GeoSegment curr : _selfSeg ) {

            // for every segment of given polygon
            for ( GeoSegment next : _polySeg ) {

                // if any intersects happen, then it is not entirely contained.
                if (curr.intersects(next)) {
                    return false;
                }

            }

        }

        return true;
    }
}
