/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial;

import static java.lang.Math.acos;
import static java.lang.Math.asin;
import static java.lang.Math.atan2;
import static java.lang.Math.cos;
import static java.lang.Math.pow;
import static java.lang.Math.sin;
import static java.lang.Math.sqrt;
import static java.lang.Math.toDegrees;
import static terraml.commons.Doubles.isGreater;
import static terraml.commons.Doubles.isNumericallyEqual;
import static terraml.commons.Doubles.isSmaller;
import static terraml.commons.Objects.isNull;
import terraml.commons.math.Angle;
import static terraml.geospatial.GeoUtils.fixLongitudeFromDegree;
import terraml.geospatial.impl.ImmutableLatlon;

// unb1337 geleceğe boykot vaat eder.
/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public final class Triangulation {

    public static class Node {

        public final Latlon source;
        public final Angle bearing;

        /**
         *
         * @param source
         * @param bearing
         */
        public Node(Latlon source, Angle bearing) {
            this.source = source;
            this.bearing = bearing;
        }
    }

    /**
     *
     * @param lat0
     * @param lon0
     * @param bearin0
     * @param lat1
     * @param lon1
     * @param bearin1
     * @return
     */
    public static double[] triangulateFromRadian(
            double lat0, double lon0, double bearin0,
            double lat1, double lon1, double bearin1
    ) {
        double _deltaX = lat1 - lat0;
        double _deltaY = lon1 - lon0;

        double _let0 = pow(sin(_deltaX / 2), 2) + cos(lat0) * cos(lat1) * pow(sin(_deltaY / 2), 2);
        _let0 = 2 * asin(sqrt(_let0));

        if (isNumericallyEqual(_let0, 0)) {
            return null;
        }

        double _deltaA = (sin(lat1) - sin(lat0) * cos(_let0)) / (sin(_let0) * cos(lat0));
        _deltaA = acos(_deltaA);

        if (Double.isNaN(_deltaA)) {
            _deltaA = 0.0d;
        }

        double _deltaB = (sin(lat0) - sin(lat1) * cos(_let0)) / (sin(_let0) * cos(lat1));
        _deltaB = acos(_deltaB);

        double _let1 = isGreater(sin(lon1 - lon0), 0) ? _deltaA : 2 * Math.PI - _deltaA;
        double _let2 = isGreater(sin(lon1 - lon0), 0) ? 2 * Math.PI - _deltaB : _deltaB;

        double _let3 = (bearin0 - _let1 + Math.PI) % (2 * Math.PI) - Math.PI;
        double _let4 = (_let2 - bearin1 + Math.PI) % (2 * Math.PI) - Math.PI;

        if (isNumericallyEqual(sin(_let3), 0) && isNumericallyEqual(sin(_let4), 0)) {
            return null;
        } else if (isSmaller(sin(_let3) * sin(_let4), 0)) {
            return null;
        }

        double _let5 = -cos(_let3) * cos(_let4) + sin(_let3) * sin(_let4) * cos(_let0);
        _let5 = acos(_let5);
        double _let6 = atan2(sin(_let0) * sin(_let3) * sin(_let4), cos(_let4) + cos(_let3) * cos(_let5));

        double lat = sin(lat0) * cos(_let6) + cos(lat0) * sin(_let6) * cos(bearin0);
        lat = asin(lat);

        double _let7 = atan2(sin(bearin0) * sin(_let6) * cos(lat0), cos(_let6) - sin(lat0) * sin(lat));
        double lon = lon0 + _let7;

        return new double[]{toDegrees(lat), fixLongitudeFromDegree(toDegrees(lon))};
    }

    /**
     *
     * @param latlon0
     * @param bearing0
     * @param latlon1
     * @param bearing1
     * @return
     */
    public static double[] triangulateFromRadian(double[] latlon0, double bearing0, double[] latlon1, double bearing1) {
        return triangulateFromRadian(
                latlon0[0], latlon0[1], bearing0,
                latlon1[0], latlon1[1], bearing1
        );
    }

    /**
     *
     * @param latlon0
     * @param fwAzimuth0
     * @param latlon1
     * @param fwAzimuth1
     * @return
     */
    public static Latlon triangulate(Latlon latlon0, Angle fwAzimuth0, Latlon latlon1, Angle fwAzimuth1) {
        final double[] triangulated = triangulateFromRadian(
                latlon0.toArrayAsRadian(), fwAzimuth0.radian,
                latlon1.toArrayAsRadian(), fwAzimuth1.radian
        );

        if (isNull(triangulated)) {
            return null;
        }

        return new ImmutableLatlon(triangulated[0], triangulated[1]);
    }

    /**
     *
     * @param node0
     * @param node1
     * @return
     */
    public static Latlon triangulate(Node node0, Node node1) {
        return triangulate(
                node0.source, node0.bearing,
                node1.source, node1.bearing
        );
    }
}
