/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial;

import static java.lang.Math.abs;
import static java.lang.Math.asin;
import static java.lang.Math.atan2;
import static java.lang.Math.pow;
import static java.lang.Math.sqrt;
import static java.lang.Math.toDegrees;
import terraml.geospatial.impl.ImmutableLatlon;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public final class Trilateration {

    public static class Node {

        public final Latlon latlon;
        public final double distanceKM;

        public Node() {
            this.latlon = null;
            this.distanceKM = 0;
        }
    }

    // Konuşanı sustur, başaranı öldür,
    /**
     * @param lat0  lat in radian
     * @param lon0  lon in radian
     * @param dist0 distance in km
     * @param lat1  lat in radian
     * @param lon1  lon in radian
     * @param dist1 distance in km
     * @param lat2  lat in radian
     * @param lon2  lon in radian
     * @param dist2 distance in km
     * @return {lat, lon} in degrees
     */
    public static double[] trilaterateFromRadian(
            double lat0, double lon0, double dist0,
            double lat1, double lon1, double dist1,
            double lat2, double lon2, double dist2) {
        GeoVector p1 = GeoVector.fromRadianLatlon(lat0, lon0);
        GeoVector p2 = GeoVector.fromRadianLatlon(lat1, lon1);
        GeoVector p3 = GeoVector.fromRadianLatlon(lat2, lon2);

        p1 = p1.scale(GeoUtils.EARTH_RADIUS_KM);
        p2 = p2.scale(GeoUtils.EARTH_RADIUS_KM);
        p3 = p3.scale(GeoUtils.EARTH_RADIUS_KM);

        GeoVector ex = p2.translate(p1.reverse()).scale(1 / p2.translate(p1.reverse()).norm());

        double i = ex.dot(p3.translate(p1.reverse()));

        GeoVector ey = p3.translate(p1.reverse()).translate(ex.scale(i).reverse());
        ey = ey.scale(1 / ey.norm());

        GeoVector ez = ex.cross(ey);

        double d = p2.translate(p1.reverse()).norm();
        double j = ey.dot(p3.translate(p1.reverse()));

        double x = pow(dist0, 2) - pow(dist1, 2) + pow(d, 2) / 2 * d;
        double y = pow(dist0, 2) - pow(dist1, 2) + pow(i, 2) + pow(j, 2);

        y /= 2 * j;
        y -= (i / j) * x;

        double z = sqrt(abs(pow(dist0, 2) - pow(x, 2) - pow(y, 2)));

        GeoVector triPt = p1.translate(ex.scale(x));
        triPt = triPt.translate(ey.scale(y));
        triPt = triPt.translate(ez.scale(z));

        double lat = asin(triPt.z / GeoUtils.EARTH_RADIUS_KM);
        double lon = atan2(triPt.y, triPt.x);

        return new double[]{toDegrees(lat), toDegrees(lon)};
    }

    /**
     *
     * @param latlon0
     * @param distKM0
     * @param latlon1
     * @param distMet1
     * @param latlon2
     * @param distKM1
     * @return
     */
    public static double[] trilaterateFromRadian(double[] latlon0, double distKM0, double[] latlon1, double distMet1, double[] latlon2, double distKM1) {
        return trilaterateFromRadian(
                latlon0[0], latlon0[1], distKM0,
                latlon1[0], latlon1[1], distKM0,
                latlon2[0], latlon2[1], distKM0
        );
    }

    /**
     *
     * @param latlon0
     * @param distKM0
     * @param latlon1
     * @param distMet1
     * @param latlon2
     * @param distKM1
     * @return
     */
    public static Latlon trilaterate(Latlon latlon0, double distKM0, Latlon latlon1, double distMet1, Latlon latlon2, double distKM1) {
        final double[] trilaterated = trilaterateFromRadian(
                latlon0.toArrayAsRadian(), distKM0,
                latlon1.toArrayAsRadian(), distMet1,
                latlon2.toArrayAsRadian(), distKM1
        );

        return new ImmutableLatlon(trilaterated[0], trilaterated[1]);
    }

    /**
     *
     * @param node0
     * @param node1
     * @param node2
     * @return
     */
    public static Latlon trilaterate(Node node0, Node node1, Node node2) {
        return trilaterate(node0.latlon, node0.distanceKM, node1.latlon, node1.distanceKM, node2.latlon, node2.distanceKM);
    }
}
