/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial;

import static java.lang.Math.abs;
import static java.lang.Math.atan2;
import static java.lang.Math.ceil;
import static java.lang.Math.cos;
import static java.lang.Math.min;
import static java.lang.Math.pow;
import static java.lang.Math.signum;
import static java.lang.Math.sin;
import static java.lang.Math.sqrt;
import static java.lang.Math.tan;
import static java.lang.Math.toRadians;
import java.util.ArrayList;
import java.util.List;
import static terraml.commons.Doubles.isSmaller;
import terraml.commons.annotation.Development;
import terraml.commons.annotation.Reference;
import static terraml.geospatial.Azimuths.northBasedAzimuth;
import static terraml.geospatial.Azimuths.northBasedFinalAzimuth;
import static terraml.geospatial.GeoUtils.fixLongitudeFromDegree;
import static terraml.geospatial.GeoUtils.lat2rad;
import static terraml.geospatial.GeoUtils.lon2deg;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public final class Zone {

    private Zone() {
    }

    /**
     *
     * @param geoPolygon
     * @return in km
     */
    public static double areaOf(GeoPolygon geoPolygon) {
        List<Latlon> vertices = geoPolygon.toList();
        int _len = vertices.size() - 1;

        double _letQ = 0.0d;
        for (int i = 0; i < _len; i++) {
            final Latlon _curr = vertices.get(i);
            final Latlon _next = vertices.get(i + 1);

            double _lat0 = _curr.getLatitude().toRadian();
            double _lat1 = _next.getLatitude().toRadian();

            double _deltaLon = toRadians(lon2deg(_next) - lon2deg(_curr));
            double _letY = tan(_deltaLon * 0.5) * (tan(_lat0 * 0.5) + tan(_lat1 * 0.5));
            double _letX = 1 + tan(_lat0 * 0.5) / tan(_lat1 * 0.5);

            _letQ += 2 * atan2(_letY, _letX);
        }

        if (isFixed(geoPolygon)) {
            _letQ = abs(_letQ) - 2 * Math.PI;
        }

        return abs(_letQ * pow(GeoUtils.EARTH_RADIUS_KM, 2));
    }

    /**
     *
     * @param geoPolygon
     * @return
     */
    private static boolean isFixed(GeoPolygon geoPolygon) {
        List<Latlon> vertices = geoPolygon.toList();

        double _fwAz = northBasedAzimuth(vertices.get(0), vertices.get(1)).degree;
        double _deltaQ = 0.0d;

        for (int i = 0; i < vertices.size() - 1; i++) {
            double _az = northBasedAzimuth(vertices.get(i), vertices.get(i + 1)).degree;
            double _faz = northBasedFinalAzimuth(vertices.get(i), vertices.get(i + 1)).degree;
            _deltaQ += fixLongitudeFromDegree(_az - _fwAz);
            _deltaQ += fixLongitudeFromDegree(_faz - _az);
            _fwAz = _faz;
        }

        double _nfaz = northBasedAzimuth(vertices.get(0), vertices.get(1)).degree;
        _deltaQ = fixLongitudeFromDegree(_nfaz - _fwAz);

        return isSmaller(abs(_deltaQ), 90.0d);
    }

    /**
     *
     * @see MATLAB.areaint
     * @param geoPolygon
     * @return
     */
    public static double areaOf2(GeoPolygon geoPolygon) {
        final List<Latlon> vertices = GeoUtils.openFormOf(geoPolygon.toList());
        final ArrayList<Double> latitudes = new ArrayList<>();
        final ArrayList<Double> longitudes = new ArrayList<>();

        vertices.stream().map((_vertex) -> {
            latitudes.add(_vertex.getLatitude().toDegree());
            return _vertex;
        }).forEachOrdered((_vertex) -> {
            longitudes.add(_vertex.getLongitude().toDegree());
        });

        return areaOf2(latitudes, longitudes);
    }

    /**
     *
     * 5.10072e14 sqM ~ Earth surface area, calculated from per unit.
     *
     * @see MATLAB 'areaint'
     * @param lats
     * @param lons
     * @return
     */
    private static double areaOf2(ArrayList<Double> lats, ArrayList<Double> lons) {
        double sum = 0;
        double prevcolat = 0;
        double prevaz = 0;
        double colat0 = 0;
        double az0 = 0;
        for (int i = 0; i < lats.size(); i++) {
            double colat = 2 * atan2(sqrt(pow(sin(lats.get(i)
                    * Math.PI / 180 / 2), 2) + cos(lats.get(i)
                            * Math.PI / 180) * pow(sin(lons.get(i)
                            * Math.PI / 180 / 2), 2)), sqrt(1 - pow(sin(lats.get(i)
                            * Math.PI / 180 / 2), 2) - cos(lats.get(i) * Math.PI / 180)
                            * pow(sin(lons.get(i) * Math.PI / 180 / 2), 2)));
            double az = 0;
            if (lats.get(i) >= 90) {
                az = 0;
            } else if (lats.get(i) <= -90) {
                az = Math.PI;
            } else {
                az = atan2(cos(lats.get(i) * Math.PI / 180)
                        * sin(lons.get(i) * Math.PI / 180),
                        sin(lats.get(i) * Math.PI / 180)) % (2 * Math.PI);
            }
            if (i == 0) {
                colat0 = colat;
                az0 = az;
            }
            if (i > 0 && i < lats.size()) {
                sum = sum + (1 - cos(prevcolat + (colat - prevcolat) / 2))
                        * Math.PI * ((abs(az - prevaz) / Math.PI) - 2
                        * ceil(((abs(az - prevaz) / Math.PI) - 1) / 2))
                        * signum(az - prevaz);
            }
            prevcolat = colat;
            prevaz = az;
        }
        sum = sum + (1 - cos(prevcolat + (colat0 - prevcolat) / 2)) * (az0 - prevaz);

        return 5.10072E14 * min(Math.abs(sum) / 4 / Math.PI, 1 - abs(sum) / 4 / Math.PI);
    }

    /**
     *
     * @param geoBoundingBox
     * @return in meters
     */
    @Development(status = Development.Status.STABLE)
    @Reference(link = "http://mathforum.org/library/drmath/view/63767.html")
    public static double areaOf(GeoBoundingBox geoBoundingBox) {
        final double _deltaSin = abs(sin(lat2rad(geoBoundingBox.getNorthEast())) - sin(lat2rad(geoBoundingBox.getSouthWest())));
        final double _deltaLon = abs(lon2deg(geoBoundingBox.getNorthEast()) - lon2deg(geoBoundingBox.getSouthWest()));
        final double _letQ = (Math.PI / 180) * pow(GeoUtils.EARTH_RADIUS_M, 2);

        final double letW = _letQ * _deltaSin * _deltaLon;

        return letW / 1000.0d;
    }

    /**
     * @param geoCircle
     * @return
     */
    @Development(status = Development.Status.STABLE)
    @Reference(link = "https://math.stackexchange.com/questions/135344/how-to-calculate-the-area-of-a-circle-given-origin-radius-on-a-sphere-ea")
    public static double areaOf(GeoCircle geoCircle) {
        final double _letQ = geoCircle.getRadius().asKilometer() / GeoUtils.EARTH_RADIUS_KM;
        final double _surf = 2 * Math.PI * Math.pow(GeoUtils.EARTH_RADIUS_KM, 2);
        final double _letW = 1 - cos(_letQ);

        return _surf * _letW;
    }
}
