/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import static terraml.commons.Doubles.isGreaterEqual;
import static terraml.commons.Doubles.isSmallerEqual;
import static terraml.commons.Objects.isNull;
import terraml.commons.math.Angle;
import terraml.commons.math.Interval;
import terraml.geospatial.DistanceNode;
import terraml.geospatial.GeoBoundingBox;
import terraml.geospatial.GeoPolyline;
import terraml.geospatial.GeoSegment;
import terraml.geospatial.GeoShapeUnit;
import terraml.geospatial.Latitude;
import terraml.geospatial.Latlon;
import terraml.geospatial.LatlonIntersection;
import terraml.geospatial.Locate;
import terraml.geospatial.Longitude;

// biz yakarsak söndüremezler.
/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public final class ImmutableGeoBoundingBox implements GeoBoundingBox, Serializable {

    public final String id;
    public final Latlon southWest;
    public final Latlon northEast;

    public ImmutableGeoBoundingBox(String id, Latlon southWest, Latlon northEast) {
        this.id = id;
        this.southWest = southWest;
        this.northEast = northEast;
    }

    /**
     * @param Latlon
     * @param Latlon
     */
    public ImmutableGeoBoundingBox(Latlon southWest, Latlon northEast) {
        final Latlon[] fixedBounds = _fixBounds(southWest, northEast);

        this.southWest = fixedBounds[0];
        this.northEast = fixedBounds[1];
        this.id = UUID.randomUUID().toString();
    }

    /**
     * @param Latitude
     * @param Longitude
     * @param Latitude
     * @param Longitude
     */
    public ImmutableGeoBoundingBox(Latitude lat0, Longitude lon0, Latitude lat1, Longitude lon1) {
        this(new ImmutableLatlon(lat0, lon0), new ImmutableLatlon(lat1, lon1));
    }

    /**
     * @param String    id
     * @param Latitude
     * @param Longitude
     * @param Latitude
     * @param Longitude
     */
    public ImmutableGeoBoundingBox(String id, Latitude lat0, Longitude lon0, Latitude lat1, Longitude lon1) {
        this(id, new ImmutableLatlon(lat0, lon0), new ImmutableLatlon(lat1, lon1));
    }

    /**
     * @param GeoBoundingBox
     */
    public ImmutableGeoBoundingBox(GeoBoundingBox geoBoundingBox) {
        this(geoBoundingBox.getSouthWest(), geoBoundingBox.getNorthEast());
    }

    /**
     * @param latlonArray
     */
    public ImmutableGeoBoundingBox(Latlon[] latlonArray) {
        this(latlonArray[0], latlonArray[1]);
    }

    /**
     * @param latlonArray
     */
    public ImmutableGeoBoundingBox(String id, Latlon[] latlonArray) {
        this(latlonArray[0], latlonArray[1]);
    }

    /**
     * @param latlons
     */
    public ImmutableGeoBoundingBox(List<Latlon> latlons) {
        this(null, latlons);
    }

    /**
     * @param id
     * @param latlons
     */
    public ImmutableGeoBoundingBox(String id, List<Latlon> latlons) {
        final List<Latlon> _bnd = Locate.locateBoundsOf(latlons);

        this.id = id;
        this.southWest = _bnd.get(0);
        this.northEast = _bnd.get(1);
    }

    /**
     * @param Latlon
     * @param Latlon
     * @return
     */
    private static Latlon[] _fixBounds(Latlon sw, Latlon ne) {
        final Interval intValX = new Interval(sw.getLatitude().toDegree(), ne.getLatitude().toDegree());
        final Interval intValY = new Interval(sw.getLongitude().toDegree(), ne.getLongitude().toDegree());

        return new Latlon[]{
            new ImmutableLatlon(intValX.left, intValY.left),
            new ImmutableLatlon(intValX.right, intValY.right)
        };
    }

    @Override
    public boolean contains(Latlon coordinate) {
        return LatlonIntersection.within(coordinate, this);
    }

    @Override
    public boolean contains(GeoBoundingBox geoBox) {
        final Latlon boxSW = geoBox.getSouthWest();
        final Latlon boxNE = geoBox.getSouthWest();

        final Latlon thisSW = this.southWest;
        final Latlon thisNE = this.northEast;

        return contains(thisSW, thisNE, boxSW, boxNE);
    }

    /**
     * @param thisSW
     * @param thisNE
     * @param thatSW
     * @param thatNE
     * @return
     */
    private boolean contains(Latlon thisSW, Latlon thisNE, Latlon thatSW, Latlon thatNE) {
        return isGreaterEqual(thatSW.lat(), thisSW.lat())
                && isSmallerEqual(thatNE.lat(), thisNE.lat())
                && isGreaterEqual(thatSW.lon(), thisSW.lon())
                && isSmallerEqual(thatNE.lon(), thisNE.lon());
    }

    /**
     * @param thisSW
     * @param thisNE
     * @param thatSW
     * @param thatNE
     * @return
     */
    private boolean disjoint(Latlon thisSW, Latlon thisNE, Latlon thatSW, Latlon thatNE) {
        final boolean situ0 = contains(thisSW, thisNE, thatSW, thatNE);
        final boolean situ1 = contains(thatSW, thatNE, thisSW, thisNE);

        return !situ0 && !situ1;
    }

    @Override
    public boolean intersects(GeoBoundingBox geoBox) {
        final Latlon boxSW = geoBox.getSouthWest();
        final Latlon boxNE = geoBox.getSouthWest();

        final Latlon thisSW = this.southWest;
        final Latlon thisNE = this.northEast;

        if (disjoint(thisSW, thisNE, boxSW, boxNE)) {
            return false;
        }

        return isGreaterEqual(boxNE.lat(), thisSW.lat())
                && isSmallerEqual(boxSW.lat(), thisNE.lat())
                && isGreaterEqual(boxNE.lon(), thisSW.lon())
                && isSmallerEqual(boxSW.lon(), thisNE.lon());
    }

    @Override
    public Collection<GeoSegment> toHeuristicSegments() {
        final List<GeoSegment> segments = new LinkedList<>();

        // avoid null-pointer-exception
        if (isNull(this.southWest) || isNull(this.northEast)) {
            return segments;
        }

        // get current corners
        final Latlon[] _bnd = getBounds();

        Latlon _sw = _bnd[0];
        Latlon _ne = _bnd[1];

        // calculate (NOT ACCURATE) other corners of this rectangle.
        Latlon _nw = new ImmutableLatlon(_ne.lat(), _sw.lon());
        Latlon _se = new ImmutableLatlon(_sw.lat(), _ne.lon());

        // create requested segments and push to the list.
        // ------------------------------------- //
        // east segment
        segments.add(new ImmutableGeoSegment(_ne, _se));

        // south segment
        segments.add(new ImmutableGeoSegment(_se, _sw));

        // west segment
        segments.add(new ImmutableGeoSegment(_sw, _nw));

        // north segment
        segments.add(new ImmutableGeoSegment(_nw, _ne));

        return segments;
    }

    @Override
    public Collection<GeoSegment> toAccurateSegments() {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /**
     * @return North-West Corner.
     */
    private Latlon nw() {
        if (nullOrEmpty()) {
            throw new IllegalStateException("Rectangle is empty");
        }

        final Latlon[] _bnd = getBounds();
        Latlon _sw = _bnd[0];
        Latlon _ne = _bnd[1];

        Latlon _nw = new ImmutableLatlon(_ne.lat(), _sw.lon());

        return _nw;
    }

    /**
     * @return SouthEast Corner
     */
    private Latlon se() {
        if (nullOrEmpty()) {
            throw new IllegalStateException("Rectangle is empty");
        }

        final Latlon[] _bnd = getBounds();
        Latlon _sw = _bnd[0];
        Latlon _ne = _bnd[1];

        Latlon _se = new ImmutableLatlon(_sw.lat(), _ne.lon());

        return _se;
    }

    @Override
    public GeoPolyline toPolylineCW() {
        if (nullOrEmpty()) {
            throw new IllegalStateException("Rectangle is empty");
        }

        List<Latlon> _crd = new ArrayList<>();

        // add coordinates order to clock-wise
        // ------------------------------------- //
        // North-East
        _crd.add(this.northEast);

        // South-East
        _crd.add(se());

        // South-West
        _crd.add(this.southWest);

        // North-West
        _crd.add(nw());

        return new ImmutableGeoPolyline(_crd);
    }

    @Override
    public GeoPolyline toPolylineCCW() {
        if (nullOrEmpty()) {
            throw new IllegalStateException("Rectangle is empty");
        }

        List<Latlon> _crd = new ArrayList<>();

        // add coordinates order to counter-clock-wise
        // ------------------------------------- //
        // North-East
        _crd.add(this.northEast);

        // North-West
        _crd.add(nw());

        // South-West
        _crd.add(this.southWest);

        // South-East
        _crd.add(se());

        return new ImmutableGeoPolyline(_crd);
    }

    @Override
    public Latlon getCenter() {
        if (nullOrEmpty()) {
            throw new IllegalStateException("Rectangle is empty");
        }

        Latlon object = new ImmutableLatlon(Locate.centerOfBounds(this.southWest, this.northEast));

        return object;
    }

    @Override
    public Latlon getCenterAccurate() {
        if (nullOrEmpty()) {
            throw new IllegalStateException("Rectangle is empty");
        }

        Latlon object = new ImmutableLatlon(Locate.centerOf(Arrays.asList(this.southWest, this.northEast)));

        return object;
    }

    @Override
    public GeoBoundingBox offset(DistanceNode distance, Angle bearing) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public Latlon getSouthWest() {
        return new ImmutableLatlon(this.southWest);
    }

    @Override
    public Latlon getNorthEast() {
        return new ImmutableLatlon(this.northEast);
    }

    @Override
    public Latlon[] toArray() {
        return new Latlon[]{getSouthWest(), getNorthEast()};
    }

    @Override
    public double[] toDoubleArray() {
        return new double[]{
            southWest.getLatitude().toDegree(),
            southWest.getLongitude().toDegree(),
            northEast.getLatitude().toDegree(),
            northEast.getLongitude().toDegree()
        };
    }

    @Override
    public ImmutableGeoBoundingBox clone() {
        return new ImmutableGeoBoundingBox(southWest, northEast);
    }

    @Override
    public GeoShapeUnit getGeoShapeUnit() {
        return GeoShapeUnit.GeoBoundingBox;
    }

    @Override
    public Latlon[] getBounds() {
        return new Latlon[]{getSouthWest(), getNorthEast()};
    }

    /**
     * @return true if null or empty. OTherwise false.
     */
    private boolean nullOrEmpty() {
        return Objects.isNull(this.southWest) || Objects.isNull(this.northEast);
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 31 * hash + Objects.hashCode(this.southWest);
        hash = 31 * hash + Objects.hashCode(this.northEast);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutableGeoBoundingBox other = (ImmutableGeoBoundingBox) obj;
        if (!Objects.equals(this.southWest, other.southWest)) {
            return false;
        }
        if (!Objects.equals(this.northEast, other.northEast)) {
            return false;
        }
        return true;
    }

    @Override
    public String getId() {
        return this.id;
    }
}
