/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.UUID;
import static terraml.commons.Doubles.isGreater;
import static terraml.commons.Doubles.isSmaller;
import terraml.commons.Objects;
import terraml.commons.tuple.Pair;
import terraml.commons.unit.DistanceUnit;
import terraml.geospatial.Distance;
import terraml.geospatial.DistanceCalculator;
import terraml.geospatial.DistanceNode;
import terraml.geospatial.GeoBoundry;
import terraml.geospatial.GeoPolyline;
import terraml.geospatial.GeoSegment;
import terraml.geospatial.GeoShapeUnit;
import terraml.geospatial.Latlon;
import terraml.geospatial.Locate;

// Çapraz Kültür avantajı.
/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public class ImmutableGeoPolyline implements GeoPolyline, Serializable {

    public final String id;
    private final List<Latlon> list;

    public ImmutableGeoPolyline(String id, List<Latlon> list) {
        this.id = id;
        this.list = list;
    }

    /**
     * @param List<Latlon>
     */
    public ImmutableGeoPolyline(List<Latlon> list) {
        this.list = list;
        this.id = UUID.randomUUID().toString();
    }

    @Override
    public double crossTrack(Latlon coordinate, boolean isMin, boolean isVincenty) {
        if (nullOrEmpty()) {
            throw new IllegalStateException("Polyline is empty.");
        }

        final Collection<GeoSegment> segments = toSegments();

        double currentMin = Double.MAX_VALUE;
        double currentMax = Double.MIN_VALUE;

        for ( GeoSegment segment : segments ) {

            // calculate track distance
            DistanceNode track = isVincenty ? segment.vinCrosstrack(coordinate) : segment.havCrosstrack(coordinate);

            // if minimum value requested.
            if (isMin) {
                // store if track is smaller then current minimum
                if (isSmaller(track.asMeter(), currentMin)) {
                    currentMin = track.asMeter();
                }
            } // if maximum value requested.
            else {
                // store if track is greater then current maximum
                if (isGreater(track.asMeter(), currentMax)) {
                    currentMax = track.asMeter();
                }
            }
        }

        return isMin ? currentMin : currentMax;
    }

    @Override
    public boolean intersects(GeoPolyline polyline) {
        if (nullOrEmpty()) {
            throw new IllegalStateException("Polyline is empty.");
        }

        final Collection<GeoSegment> primary = toSegments();
        final Collection<GeoSegment> secondary;

        /**
         * // fix casting
         * if ( polyline instanceof GeoPolyline ) {
         * // export segments
         * secondary = polyline.toSegments();
         * } else {
         * // export segments
         * secondary = new TRSPolylineImpl(polyline).toSegments();
         * }
         */
        secondary = polyline.toSegments();

        if (secondary.isEmpty()) {
            return false;
        }

        // equals 2xfor
        return primary.stream()
                .anyMatch((current) -> (secondary.stream()
                                        .anyMatch((next) -> (current.intersects(next)))));
    }

    @Override
    public Collection<Latlon> intersection(GeoPolyline polyline, DistanceNode tolerance) {
        if (nullOrEmpty()) {
            throw new IllegalStateException("Polyline is empty.");
        }

        final Collection<Latlon> tmpList = new LinkedList<>();

        final Collection<GeoSegment> primary = toSegments();
        final Collection<GeoSegment> secondary = polyline.toSegments();

        if (secondary.isEmpty()) {
            return new ArrayList<>();
        }

        // for every segment of this.
        primary.forEach((current) -> {

            // if intersection found
            secondary.stream()
                    .map((next) -> current.intersection(next, tolerance))
                    .filter((inter) -> (inter != null))
                    .forEachOrdered((inter) -> {

                        // then push to the list.
                        tmpList.add(new ImmutableLatlon(inter));

                    });

        });

        return tmpList;
    }

    @Override
    public boolean isOn(Latlon point, DistanceNode tolerance) {
        if (nullOrEmpty()) {
            throw new IllegalStateException("Polyline is empty.");
        }

        final Collection<GeoSegment> segments = toSegments();

        return segments.stream().anyMatch((segment) -> (segment.isOn(point, tolerance)));
    }

    @Override
    public boolean isOn(Latlon point, DistanceCalculator calculator, DistanceNode tolerance) {
        if (nullOrEmpty()) {
            throw new IllegalStateException("Polyline is empty.");
        }

        final Collection<GeoSegment> segments = toSegments();

        return segments.stream().anyMatch((segment) -> (segment.isOn(point, tolerance, calculator)));
    }

    @Override
    public Pair<Latlon, DistanceNode> closestOf(Latlon coord, DistanceCalculator calculator) {
        if (nullOrEmpty()) {
            throw new IllegalStateException("Polyline is empty.");
        }

        final Collection<GeoSegment> segments = toSegments();

        DistanceNode distance = new DistanceNode(DistanceUnit.METER, Double.MAX_VALUE);
        Latlon closest = null;

        // check for every segment of this.
        for ( GeoSegment segment : segments ) {

            // offsetted closest for current segment
            Latlon currentClosest = segment.closestOf(coord, calculator);

            // if found closest is null (in the beginning 'closest' is null)
            if (Objects.isNull(closest)) {

                // declare new values
                closest = currentClosest;
                distance = calculator.distanceOf(coord, closest);

            } else {

                // offsetted closest distance for current segment
                DistanceNode trackDistance = calculator.distanceOf(coord, currentClosest);

                // if current closest is closer than before
                if (isSmaller(trackDistance.asMeter(), distance.asMeter())) {

                    // declare new values
                    closest = currentClosest;
                    distance = trackDistance;
                }

            }
        }

        return new Pair<>(closest, distance);
    }

    @Override
    public Latlon find(DistanceNode node) {
        if (nullOrEmpty()) {
            throw new IllegalStateException("Polyline cannot be offset. It is empty");
        }

        final Collection<GeoSegment> segments = toSegments();
        final DistanceCalculator calculator = new Distance.Vincenty();

        DistanceNode tracker = node;

        // check for every segment of this.
        for ( GeoSegment segment : segments ) {

            // offsetted track distance.
            final DistanceNode distance = segment.distance(calculator);

            // if requested coordinate is on current segment
            if (tracker.asMeter() < distance.asMeter()) {

                // return requested.
                return new ImmutableLatlon(segment.offset(tracker).getSource());

            } else {

                // before going next segment, fix distance.
                double val = tracker.asMeter() - distance.asMeter();
                tracker = new DistanceNode(DistanceUnit.METER, val);

            }
        }

        return null;
    }

    // günlüklerim aylık olmuş.
    
    @Override
    public Latlon centroid() {
        if (nullOrEmpty()) {
            throw new IllegalStateException("Empty Polyline.");
        }

        final List<Latlon> wildcard = new ArrayList<>();

        for ( Latlon coordinate : list ) {
            wildcard.add(coordinate);
        }

        return new ImmutableLatlon(Locate.centerOf(wildcard));
    }

    @Override
    public Collection<GeoSegment> toSegments() {
        Collection<GeoSegment> segments = new LinkedList<>();

        final int len = list.size();

        for ( int i = 0; i < len; i++ ) {

            // index of segment's target
            final int next = i + 1;

            // avoid OutOfIndex
            if (next < len) {

                // init source point of segment
                Latlon currentSrc = list.get(i);

                // init target point of segment
                Latlon currentTar = list.get(i + 1);

                // push to the list
                segments.add(new ImmutableGeoSegment(currentSrc, currentTar));
            }
        }

        return segments;
    }

    @Override
    public Latlon[] toArray() {
        if (Objects.isNull(list)) {
            return new Latlon[]{};
        }

        final int len = list.size();
        final Latlon[] array = new Latlon[len];

        for ( int i = 0; i < len; i++ ) {
            array[i] = list.get(i);
        }

        return array;
    }

    @Override
    public List<Latlon> toList() {
        return new ArrayList<>(list);
    }

    @Override
    public GeoPolyline clone() {
        return new ImmutableGeoPolyline(list);
    }

    @Override
    public GeoShapeUnit getGeoShapeUnit() {
        return GeoShapeUnit.GeoPolyline;
    }

    @Override
    public Latlon[] getBounds() {
        if (nullOrEmpty()) {
            return new ImmutableLatlon[]{};
        }
        
        final GeoBoundry geoBoundry = Locate.boundsOf(list);

        return new Latlon[]{
            new ImmutableLatlon(geoBoundry.lowerBound),
            new ImmutableLatlon(geoBoundry.upperBound)
        };
    }

    @Override
    public String getId() {
        return id;
    }

    /**
     * @return true if null or empty. Otherwise false.
     */
    private boolean nullOrEmpty() {
        return this.list == null || this.list.isEmpty();
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 97 * hash + java.util.Objects.hashCode(this.list);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutableGeoPolyline other = (ImmutableGeoPolyline) obj;
        if (!java.util.Objects.equals(this.list, other.list)) {
            return false;
        }
        return true;
    }
}
