/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial.impl;

import java.io.Serializable;
import static terraml.commons.Doubles.isGreater;
import static terraml.commons.Doubles.isSmaller;
import terraml.commons.math.Angle;
import terraml.geospatial.Latitude;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public final class ImmutableLatitude implements Latitude, Serializable  {

    public final double lat;

    /**
     * @param double 
     */
    public ImmutableLatitude(double lat) {
        if (!_isValid(lat)) {
            throw new IllegalArgumentException("Given" + lat + 
                    " angle must be in range. Hint: {@MIN_LATITUDE, @MAX_LATITUDE}");
        }

        this.lat = lat;
    }

    /**
     * @param Angle 
     */
    public ImmutableLatitude(Angle _angle) {
        this(_angle.degree);
    }

    /**
     * @param Latitude 
     */
    public ImmutableLatitude(Latitude latitude) {
        this(latitude.toDegree());
    }

    /**
     * @param double
     * @return 
     */
    private static boolean _isValid(double _ang) {
        return isGreater(_ang, MIN_LATITUDE) && isSmaller(_ang, MAX_LATITUDE);
    }

    @Override
    public Latitude translate(Latitude latitude) {
        return new ImmutableLatitude(lat + latitude.toDegree());
    }

    @Override
    public Latitude translate(double degree) {
        return new ImmutableLatitude(lat + degree);
    }

    @Override
    public Latitude scale(Latitude latitude) {
        return new ImmutableLatitude(lat * latitude.toDegree());
    }

    @Override
    public Latitude scale(double degree) {
        return new ImmutableLatitude(lat * degree);
    }

    @Override
    public double sin() {
        return Math.sin(toRadian());
    }

    @Override
    public double cos() {
        return Math.cos(toRadian());
    }

    @Override
    public double tan() {
        return Math.tan(toRadian());
    }

    @Override
    public double acos() {
        return Math.acos(toRadian());
    }

    @Override
    public double asin() {
        return Math.asin(toRadian());
    }

    @Override
    public double atan() {
        return Math.atan(toRadian());
    }

    @Override
    public Angle toAngle() {
        return Angle.fromDegree(lat);
    }

    @Override
    public double toDegree() {
        return this.lat;
    }

    @Override
    public double toRadian() {
        return Math.toRadians(lat);
    }

    @Override
    public int compareTo(Latitude latitude) {
        return Double.compare(lat, latitude.toDegree());
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 29 * hash + (int) (Double.doubleToLongBits(this.lat) ^ (Double.doubleToLongBits(this.lat) >>> 32));
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutableLatitude other = (ImmutableLatitude) obj;
        if (Double.doubleToLongBits(this.lat) != Double.doubleToLongBits(other.lat)) {
            return false;
        }
        return true;
    }
}
