/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial.impl;

import java.io.Serializable;
import java.util.Objects;
import java.util.UUID;
import terraml.commons.math.Angle;
import terraml.commons.tuple.LatlonEntry;
import terraml.commons.unit.DirectionUnit;
import terraml.commons.unit.DistanceUnit;
import static terraml.geospatial.Azimuths.northBasedAzimuth;
import static terraml.geospatial.Azimuths.northBasedFinalAzimuth;
import terraml.geospatial.Distance;
import terraml.geospatial.DistanceCalculator;
import terraml.geospatial.DistanceNode;
import terraml.geospatial.GeoShapeUnit;
import terraml.geospatial.GeoUtils;
import terraml.geospatial.GeoVector;
import terraml.geospatial.Latitude;
import terraml.geospatial.Latlon;
import terraml.geospatial.Longitude;
import terraml.geospatial.Offset;

// Sorma, zor soruları bilemem.
/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public final class ImmutableLatlon implements Latlon, Serializable {

    public final String id;
    public final Latitude latitude;
    public final Longitude longitude;

    /**
     * @param id
     * @param latitude
     * @param longitude
     */
    public ImmutableLatlon(String id, Latitude latitude, Longitude longitude) {
        this.id = id;
        this.latitude = latitude;
        this.longitude = longitude;
    }

    /**
     * @param Latitude
     * @param Longitude
     */
    public ImmutableLatlon(Latitude lat, Longitude lon) {
        this.latitude = lat;
        this.longitude = lon;
        this.id = UUID.randomUUID().toString();
    }

    /**
     * @param double
     * @param double
     */
    public ImmutableLatlon(double lat, double lon) {
        this(new ImmutableLatitude(lat), new ImmutableLongitude(lon));
    }

    /**
     * @param id
     * @param lat
     * @param lon
     */
    public ImmutableLatlon(String id, double lat, double lon) {
        this(id, new ImmutableLatitude(lat), new ImmutableLongitude(lon));
    }

    /**
     * @param Latlon
     */
    public ImmutableLatlon(Latlon latlon) {
        this(latlon.getLatitude(), latlon.getLongitude());
    }

    /**
     * @param id
     * @param latlon
     */
    public ImmutableLatlon(String id, Latlon latlon) {
        this(id, latlon.getLatitude(), latlon.getLongitude());
    }

    /**
     * @param LatlonEntry
     */
    public ImmutableLatlon(LatlonEntry entry) {
        this(entry.lat(), entry.lon());
    }

    /**
     * @param entry
     */
    public ImmutableLatlon(String id, LatlonEntry entry) {
        this(entry.lat(), entry.lon());
    }

    // dert adamdan başka mahlukatta olmaz.
    @Override
    public DistanceNode distanceTo(Latlon latlon, DistanceCalculator calculator) {
        DistanceCalculator calc;

        if (calculator == null) {
            calc = new Distance.Vincenty();
        } else {
            calc = calculator;
        }

        return calc.distanceOf(this, latlon);
    }

    @Override
    public DirectionUnit directionTo(Latlon latlon) {
        return GeoUtils.direction(this, latlon);
    }

    @Override
    public Angle bearingTo(Latlon latlon) {
        return GeoUtils.bearingCCW(this, latlon);
    }

    @Override
    public Angle azimuthTo(Latlon latlon) {
        return northBasedAzimuth(this, latlon);
    }

    @Override
    public Angle finalAzimuthTo(Latlon latlon) {
        return northBasedFinalAzimuth(this, latlon);
    }

    @Override
    public Latlon offset(DistanceNode distanceNode, Angle northBasedAzimuth) {
        double distance;

        if (distanceNode.unit.equals(DistanceUnit.METER)) {
            distance = distanceNode.distance;
        } else {
            distance = DistanceUnit.KILOMETER.toMeter(distanceNode.distance);
        }

        return Offset.destinationOf(this, distance, northBasedAzimuth);
    }

    @Override
    public GeoVector toVector() {
        return GeoVector.fromLatlon(this);
    }

    @Override
    public Latitude getLatitude() {
        return new ImmutableLatitude(this.latitude);
    }

    @Override
    public Longitude getLongitude() {
        return new ImmutableLongitude(this.longitude);
    }

    @Override
    public double[] toArray() {
        return new double[]{
            latitude.toDegree(),
            longitude.toDegree()
        };
    }

    @Override
    public double[] toArrayAsRadian() {
        return new double[]{
            latitude.toRadian(),
            longitude.toRadian()
        };
    }

    @Override
    public Latlon[] getBounds() {
        return new Latlon[]{
            new ImmutableLatlon(latitude.toDegree(), latitude.toDegree()),
            new ImmutableLatlon(longitude.toDegree(), longitude.toDegree())
        };
    }

    @Override
    public double lat() {
        return getLatitude().toDegree();
    }

    @Override
    public double lon() {
        return getLongitude().toDegree();
    }

    @Override
    public GeoShapeUnit getGeoShapeUnit() {
        return GeoShapeUnit.Latlon;
    }

    @Override
    public ImmutableLatlon clone() {
        return new ImmutableLatlon(latitude, longitude);
    }

    @Override
    public String getId() {
        return id;
    }

    @Override
    public int hashCode() {
        int hash = 3;
        hash = 67 * hash + Objects.hashCode(this.latitude);
        hash = 67 * hash + Objects.hashCode(this.longitude);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutableLatlon other = (ImmutableLatlon) obj;
        if (!Objects.equals(this.latitude, other.latitude)) {
            return false;
        }
        if (!Objects.equals(this.longitude, other.longitude)) {
            return false;
        }
        return true;
    }
}
