/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * This file is part of terraml-geospatial  project.
 *
 * This file incorporates work covered by
 * the following copyright and permission notices:
 *
 * Copyright (C) 2018 Terra Software Informatics LLC. | info [at] terrayazilim [dot] com [dot] tr
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package terraml.geospatial.impl;

import java.io.Serializable;
import static terraml.commons.Doubles.isGreater;
import static terraml.commons.Doubles.isSmaller;
import terraml.commons.math.Angle;
import terraml.geospatial.Longitude;

/**
 * @author M.Çağrı Tepebaşılı - cagritepebasili [at] protonmail [dot] com
 * @version 1.0.0-SNAPSHOT
 */
public final class ImmutableLongitude implements Longitude, Serializable {

    public static double MIN_LONGITUDE = Double.valueOf("-180.0000");
    public static double MAX_LONGITUDE = Double.valueOf("180.0000");

    public final double lon;

    /**
     * @param double 
     */
    public ImmutableLongitude(double lon) {
        if (!_isValid(lon)) {
            throw new IllegalArgumentException("Given" + lon + 
                    " angle must be in range. Hint: {@MIN_LONGITUDE, @MAX_LONGITUDE}");
        }

        this.lon = lon;
    }

    /**
     * @param Angle 
     */
    public ImmutableLongitude(Angle angle) {
        this(angle.degree);
    }

    /**
     * @param Longitude 
     */
    public ImmutableLongitude(Longitude longitude) {
        this(longitude.toDegree());
    }

    /**
     * @param double
     * @return 
     */
    private static boolean _isValid(double degree) {
        return isSmaller(degree, MAX_LONGITUDE) && isGreater(degree, MIN_LONGITUDE);
    }

    @Override
    public Longitude translate(Longitude longitude) {
        return new ImmutableLongitude(lon + longitude.toDegree());
    }

    @Override
    public Longitude translate(double degree) {
        return new ImmutableLongitude(lon + degree);
    }

    @Override
    public Longitude scale(Longitude longitude) {
        return new ImmutableLongitude(lon * longitude.toDegree());
    }

    @Override
    public Longitude scale(double degree) {
        return new ImmutableLongitude(lon * degree);
    }

    @Override
    public double sin() {
        return Math.sin(toRadian());
    }

    @Override
    public double cos() {
        return Math.cos(toRadian());
    }

    @Override
    public double tan() {
        return Math.tan(toRadian());
    }

    @Override
    public double acos() {
        return Math.acos(toRadian());
    }

    @Override
    public double asin() {
        return Math.asin(toRadian());
    }

    @Override
    public double atan() {
        return Math.atan(toRadian());
    }

    @Override
    public Angle toAngle() {
        return Angle.fromDegree(lon);
    }

    @Override
    public double toDegree() {
        return this.lon;
    }

    @Override
    public double toRadian() {
        return Math.toRadians(lon);
    }

    @Override
    public int compareTo(Longitude longitude) {
        return Double.compare(lon, longitude.toDegree());
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 37 * hash + (int) (Double.doubleToLongBits(this.lon) ^ (Double.doubleToLongBits(this.lon) >>> 32));
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final ImmutableLongitude other = (ImmutableLongitude) obj;
        if (Double.doubleToLongBits(this.lon) != Double.doubleToLongBits(other.lon)) {
            return false;
        }
        return true;
    }
}
