package online.xcodes.ip;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.ImmutableList;
import lombok.Data;
import lombok.SneakyThrows;
import org.apache.commons.lang3.StringUtils;

import java.net.Inet4Address;
import java.net.InetAddress;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 1. https://www.cz88.net/help?id=free
 * 2. https://zhuanlan.zhihu.com/p/360624952
 */
public class IPLocationUtil {

    public static final String INIT_QQWRY_ERROR = "初始化qqwry失败";

    @Data
    public static class IPLocation {
        private String country;
        private String area;
        private String isp;
        private String province;
        private String city;
    }

    private static final String COUNTRY_CHINA = "中国";

    private static final List<String> PROVINCES_CHINA = ImmutableList.of("黑龙江省", "辽宁省", "吉林省", "河北省", "河南省", "湖北省", "湖南省", "山东省", "山西省", "陕西省",
                  "安徽省", "浙江省", "江苏省", "福建省", "广东省", "海南省", "四川省", "云南省", "贵州省", "青海省", "甘肃省",
                  "江西省", "台湾省", "内蒙古", "宁夏", "新疆", "西藏", "广西", "北京市", "上海市", "天津市", "重庆市", "香港", "澳门");


    private static final List<String> ISPS_CHINA = ImmutableList.of( "联通","移动","铁通","电信","长城", "聚友");

    private static Cache<String, IPLocation> cache = CacheBuilder.newBuilder().maximumSize(200).expireAfterWrite(10, TimeUnit.MINUTES).build();

    private static QQWry qqwry;


    @SneakyThrows
    public static IPLocation search(String ip) {
        if( StringUtils.isBlank(ip) ){return new IPLocation();}
        if (! (InetAddress.getByName(ip) instanceof Inet4Address)) {
            return new IPLocation();
        }
        return cache.get(ip, () -> convert(getQQWry().findIP(ip)));
    }

    private static IPLocation convert(QQWry.IPZone ipZone){
        IPLocation ipLocation = new IPLocation();
        ipLocation.setArea(ipZone.getMainInfo());
        if("纯真网络".equals(ipZone.getMainInfo())){
            //255.255.255.0   255.255.255.255 纯真网络 2024年01月10日IP数据
            ipLocation.setCountry("IANA");
            ipLocation.setProvince("");
            ipLocation.setCity("");
            ipLocation.setIsp("");
            ipLocation.setArea("保留地址");
            return ipLocation;
        }

        //处理省市
        String province = getProvince(ipZone.getMainInfo());
        if(province!=null){
            ipLocation.setCountry(COUNTRY_CHINA);
            ipLocation.setProvince(province);
            ipLocation.setCity(ipZone.getMainInfo().replace(province,""));
        }else{
            ipLocation.setCountry(ipZone.getMainInfo());
            ipLocation.setProvince("");
            ipLocation.setCity("");
        }

        //处理ISP
        String isp = getISP(ipZone.getSubInfo());
        if(isp!=null){
            ipLocation.setIsp(isp);
        }else if("CZ88.NET".equals(ipZone.getSubInfo()) || "纯真网络".equals(ipZone.getSubInfo())){
            ipLocation.setIsp("");
        }else{
            ipLocation.setIsp(ipZone.getSubInfo());
        }
        return ipLocation;
    }

    private static String getProvince(String mainInfo){
        for (String province : PROVINCES_CHINA) {
            if(mainInfo.contains(province)){
                return province;
            }
        }
        return null;
    }

    private static String getISP(String subInfo){
        for (String isp : ISPS_CHINA) {
            if(subInfo.contains(isp)){
                return isp;
            }
        }
        return null;
    }



    private static QQWry getQQWry(){
        if(qqwry != null){
            return qqwry;
        }
        synchronized (IPLocationUtil.class){
            if(qqwry != null){
                return qqwry;
            }
            try {
                qqwry = new QQWry();
                return qqwry;
            } catch (Exception e) {
                throw new RuntimeException(INIT_QQWRY_ERROR,e);
            }
        }
    }





}
