package org.accidia.echo.resources;

import com.codahale.metrics.annotation.ExceptionMetered;
import com.codahale.metrics.annotation.Metered;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import org.accidia.echo.EchoContext;
import org.accidia.echo.Constants;
import org.accidia.echo.protos.Protos;
import org.accidia.echo.services.ITenantService;
import org.accidia.protoserver.misc.AsyncResponses;
import org.accidia.protoserver.misc.MediaTypes;
import org.glassfish.jersey.server.ManagedAsync;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.*;
import javax.ws.rs.container.AsyncResponse;
import javax.ws.rs.container.Suspended;
import javax.ws.rs.core.MediaType;
import java.util.List;
import java.util.concurrent.TimeUnit;

@Path("/v1/tenant/")
public class TenantResource {
    private static final Logger logger = LoggerFactory.getLogger(TenantResource.class);

    private final ITenantService tenantService = EchoContext.INSTANCE.getInjector()
            .getInstance(ITenantService.class);

    private long requestTimeoutInSeconds = EchoContext.INSTANCE
            .getConfiguration().getConfig().getLong(Constants.CONFIG_KEY__RESOURCE_TIMEOUT_SECONDS);

    @GET
    @Metered @ExceptionMetered
    @Path("/")
    @Produces({MediaTypes.APPLICATION_PROTOBUF + ";qs=0.5", MediaType.APPLICATION_JSON})
    @ManagedAsync
    public void getTenantes(@Suspended final AsyncResponse asyncResponse) {
        logger.debug("getDefaultTenants()");

        // then add timeout and completion handler
        AsyncResponses.addTimeoutHandler(asyncResponse, this.requestTimeoutInSeconds, TimeUnit.SECONDS);
        AsyncResponses.addCompletionCallback(asyncResponse);

        doGetTenants(asyncResponse);
    }

    @POST
    @Metered
    @ExceptionMetered
    @Path("/")
    @Consumes({MediaType.APPLICATION_JSON})
    @ManagedAsync
    public void postTenant(@Suspended final AsyncResponse asyncResponse,
                           final String tenantJson) {
        logger.debug("postTenant()");
        Preconditions.checkArgument(!Strings.isNullOrEmpty(tenantJson), "null/empty tenantJson");

        AsyncResponses.addTimeoutHandler(asyncResponse, this.requestTimeoutInSeconds, TimeUnit.SECONDS);
        AsyncResponses.addCompletionCallback(asyncResponse);

        // TODO
//        doPostTenant(tenant);
    }

    protected void doGetTenants(final AsyncResponse asyncResponse) {
        final List<Protos.Tenant> allTenants = this.tenantService.getAllTenants();
        asyncResponse.resume(allTenants);
    }
}

